import 'package:flutter/material.dart';
import 'package:nfc_manager/nfc_manager.dart';
import 'dart:typed_data';
import 'google_sheets_api.dart';

class InscriptionPage extends StatefulWidget {
  final String? prefilledMac; // valeur scannée
  final String scanType;      // "QR" ou "NFC"

  const InscriptionPage({super.key, this.prefilledMac, required this.scanType});

  @override
  State<InscriptionPage> createState() => _InscriptionPageState();
}

class _InscriptionPageState extends State<InscriptionPage> {
  final _formKey = GlobalKey<FormState>();
  final TextEditingController _numController = TextEditingController();
  final TextEditingController _firstController = TextEditingController();
  final TextEditingController _lastController = TextEditingController();
  final TextEditingController _macController = TextEditingController();
  final TextEditingController _qrController = TextEditingController();
  bool _hasNfc = false;
  bool _isScanning = false;

  final Color _primaryColor = const Color(0xFF4361EE);
  final Color _successColor = const Color(0xFF06D6A0);

  @override
  void initState() {
    super.initState();
    // Pré-remplir selon le type de scan
    if (widget.prefilledMac != null) {
      if (widget.scanType == "QR") {
        _qrController.text = widget.prefilledMac!;
      } else {
        _macController.text = widget.prefilledMac!;
        _hasNfc = true;
      }
    }
  }

  @override
  void dispose() {
    _numController.dispose();
    _firstController.dispose();
    _lastController.dispose();
    _macController.dispose();
    _qrController.dispose();
    super.dispose();
  }

  String _bytesToUid(dynamic raw) {
    try {
      final bytes = raw is Uint8List ? raw : Uint8List.fromList(List<int>.from(raw));
      return bytes.map((b) => b.toRadixString(16).padLeft(2, '0').toUpperCase()).join('');
    } catch (_) {
      return raw.toString();
    }
  }

  Future<void> _scanNfcAndFill() async {
    setState(() => _isScanning = true);
    bool available = await NfcManager.instance.isAvailable();
    if (!available) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(
          content: Text('NFC non disponible'),
          backgroundColor: Colors.orange,
        ),
      );
      setState(() => _isScanning = false);
      return;
    }

    NfcManager.instance.startSession(
      pollingOptions: {NfcPollingOption.iso14443, NfcPollingOption.iso15693},
      onDiscovered: (NfcTag tag) async {
        try {
          dynamic data;
          try { data = (tag as dynamic).data; } catch (_) { data = null; }
          String uid = '';
          try {
            final id = (data is Map && data.containsKey('id')) ? data['id'] : (tag as dynamic).id;
            uid = _bytesToUid(id);
          } catch (_) {
            uid = _bytesToUid((tag as dynamic).id);
          }
          if (uid.isNotEmpty) {
            _macController.text = uid;
            _hasNfc = true;
          }
        } catch (_) {}
        await NfcManager.instance.stopSession();
        setState(() => _isScanning = false);
      },
    );
  }

  Future<void> _submit() async {
    if (!_formKey.currentState!.validate()) return;

    final studentNumber = _numController.text.trim();
    final api = GoogleSheetsApi();

    final rowIndex = await api.findRowByStudentNumber(studentNumber);

    if (rowIndex != null) {
      //  Étudiant EXISTE → mise à jour
      final ok = await api.updateStudentFields(
        rowIndex: rowIndex,
        nfc: _macController.text.trim().isEmpty ? null : _macController.text.trim(),
        qr: _qrController.text.trim().isEmpty ? null : _qrController.text.trim(),
      );

      if (ok) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('✅ Étudiant mis à jour')),
        );
        Navigator.pop(context);
      }
      return;
    }

    //  Étudiant n’existe pas → création
    final row = [
      studentNumber,
      _firstController.text.trim(),
      _lastController.text.trim(),
      _macController.text.trim(),
      _qrController.text.trim(),
      DateTime.now().toIso8601String(),
      DateTime.now().toIso8601String(),
    ];

    final ok = await api.appendData(row);

    if (ok) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('✅ Étudiant enregistré')),
      );
      Navigator.pop(context);
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: const Color(0xFFF8F9FA),
      appBar: AppBar(
        title: const Text('Inscription Étudiant', style: TextStyle(fontWeight: FontWeight.bold)),
        backgroundColor: _primaryColor,
      ),
      body: SingleChildScrollView(
        padding: const EdgeInsets.all(20),
        child: Form(
          key: _formKey,
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              _buildTextField(controller: _numController, label: "N° Étudiant", icon: Icons.badge_rounded, mandatory: true),
              const SizedBox(height: 16),
              _buildTextField(controller: _firstController, label: "Prénom", icon: Icons.person_rounded, mandatory: true),
              const SizedBox(height: 16),
              _buildTextField(controller: _lastController, label: "Nom", icon: Icons.person_outline_rounded, mandatory: true),
              const SizedBox(height: 16),
              _buildTextField(controller: _macController, label: "NFC UID", icon: Icons.nfc_rounded, mandatory: false),
              const SizedBox(height: 16),
              _buildTextField(controller: _qrController, label: "QR Code", icon: Icons.qr_code_rounded, mandatory: false),
              const SizedBox(height: 32),
              ElevatedButton.icon(
                onPressed: _submit,
                icon: const Icon(Icons.save_rounded),
                label: const Text("Enregistrer l'étudiant"),
              ),
              const SizedBox(height: 16),
              ElevatedButton.icon(
                onPressed: _scanNfcAndFill,
                icon: _isScanning
                    ? const SizedBox(
                  width: 16,
                  height: 16,
                  child: CircularProgressIndicator(strokeWidth: 2, color: Colors.white),
                )
                    : const Icon(Icons.nfc_rounded),
                label: Text(_isScanning ? 'Scan...' : 'Scanner NFC'),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildTextField({
    required TextEditingController controller,
    required String label,
    required IconData icon,
    bool mandatory = true,
  }) {
    return TextFormField(
      controller: controller,
      decoration: InputDecoration(
        labelText: label,
        prefixIcon: Icon(icon, color: _primaryColor),
        border: OutlineInputBorder(borderRadius: BorderRadius.circular(8)),
      ),
      validator: mandatory
          ? (v) => (v == null || v.trim().isEmpty) ? 'Veuillez remplir ce champ' : null
          : null,
    );
  }
}