import 'package:flutter/material.dart';
import 'package:nfc_manager/nfc_manager.dart';
import 'dart:typed_data';
import 'google_sheets_api.dart';
// helpers for NFC bytes -> hex


class InscriptionPage extends StatefulWidget {
  final String? prefilledMac; // valeur scannée
  final String scanType;      // "QR" ou "NFC"

  const InscriptionPage({super.key, this.prefilledMac, required this.scanType});

  @override
  State<InscriptionPage> createState() => _InscriptionPageState();
}

class _InscriptionPageState extends State<InscriptionPage> {
  final _formKey = GlobalKey<FormState>();
  final TextEditingController _numController = TextEditingController();
  final TextEditingController _firstController = TextEditingController();
  final TextEditingController _lastController = TextEditingController();
  final TextEditingController _macController = TextEditingController();
  final TextEditingController _qrController = TextEditingController();
  bool _hasNfc = false;
  bool _isScanning = false;

  final Color _primaryColor = const Color(0xFF4361EE);
  final Color _successColor = const Color(0xFF06D6A0);

  @override
  void initState() {
    super.initState();
    // Pré-remplir selon le type de scan
    if (widget.prefilledMac != null) {
      if (widget.scanType == "QR") {
        _qrController.text = widget.prefilledMac!;
      } else {
        _macController.text = widget.prefilledMac!;
        _hasNfc = true;
      }
    }
  }

  Future<void> _scanNfcAndFill() async {
    setState(() => _isScanning = true);
    bool available = await NfcManager.instance.isAvailable();
    if (!available) {
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(
          content: const Text('NFC non disponible'),
          backgroundColor: Colors.orange,
          behavior: SnackBarBehavior.floating,
        ),
      );
      setState(() => _isScanning = false);
      return;
    }

    NfcManager.instance.startSession(
      pollingOptions: {NfcPollingOption.iso14443, NfcPollingOption.iso15693},
      onDiscovered: (NfcTag tag) async {
        try {
          dynamic data;
          try {
            data = (tag as dynamic).data;
          } catch (_) {
            data = null;
          }
          String uid = '';
          try {
            final id = (data is Map && data.containsKey('id')) ? data['id'] : (tag as dynamic).id;
            uid = _bytesToUid(id);
          } catch (_) {
            try {
              uid = _bytesToUid((tag as dynamic).id);
            } catch (_) {
              uid = '';
            }
          }
          if (uid.isNotEmpty) {
            setState(() {
              _macController.text = uid;
              _hasNfc = true;
            });
          }
        } catch (e) {
          ScaffoldMessenger.of(context).showSnackBar(
            SnackBar(
              content: Text('Erreur lecture NFC: $e'),
              backgroundColor: Colors.red,
              behavior: SnackBarBehavior.floating,
            ),
          );
        }
        await NfcManager.instance.stopSession();
        setState(() => _isScanning = false);
      },
    );
  }

  @override
  void dispose() {
    _numController.dispose();
    _firstController.dispose();
    _lastController.dispose();
    _macController.dispose();
    _qrController.dispose();
    super.dispose();
  }

  Future<void> _submit() async {
    if (!_formKey.currentState!.validate()) return;

    final studentNumber = _numController.text.trim();
    final api = GoogleSheetsApi();

    final rowIndex = await api.findRowByStudentNumber(studentNumber);

    if (rowIndex != null) {
      //  Étudiant EXISTE → mise à jour
      final ok = await api.updateStudentFields(
        rowIndex: rowIndex,
        nfc: _macController.text.trim().isEmpty ? null : _macController.text.trim(),
        qr: _qrController.text.trim().isEmpty ? null : _qrController.text.trim(),
      );

      if (ok) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('✅ Étudiant mis à jour')),
        );
        Navigator.pop(context);
      }
      return;
    }

    //  Étudiant n’existe pas → création
    final row = [
      studentNumber,
      _firstController.text.trim(),
      _lastController.text.trim(),
      _macController.text.trim(),
      _qrController.text.trim(),
      DateTime.now().toIso8601String(),
      DateTime.now().toIso8601String(),
    ];

    final ok = await api.appendData(row);

    if (ok) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('✅ Étudiant enregistré')),
      );
      Navigator.pop(context);
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: const Color(0xFFF6F8FA),
      appBar: AppBar(
        title: const Text(
          'Inscription Étudiant',
          style: TextStyle(fontWeight: FontWeight.bold),
        ),
        backgroundColor: _primaryColor,
        foregroundColor: Colors.white,
        elevation: 0,
      ),
      body: SingleChildScrollView(
        padding: const EdgeInsets.all(12),
        child: Card(
          elevation: 2,
          shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
          child: Padding(
            padding: const EdgeInsets.all(16),
            child: Form(
              key: _formKey,
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.stretch,
                children: [
                  // En-tête du formulaire
                  Container(
                    padding: const EdgeInsets.all(16),
                    decoration: BoxDecoration(
                      color: _primaryColor.withOpacity(0.1),
                      borderRadius: BorderRadius.circular(12),
                    ),
                    child: Row(
                      children: [
                        Icon(Icons.person_add_alt_1_rounded, color: _primaryColor),
                        const SizedBox(width: 12),
                        const Expanded(
                          child: Text(
                            'Informations étudiant',
                            style: TextStyle(
                              fontSize: 16,
                              fontWeight: FontWeight.bold,
                            ),
                          ),
                        ),
                      ],
                    ),
                  ),

                  const SizedBox(height: 16),

                  // Champs du formulaire
                  _buildTextField(
                    controller: _numController,
                    label: "N° Étudiant",
                    icon: Icons.badge_rounded,
                    mandatory: true,
                  ),

                  const SizedBox(height: 16),

                  _buildTextField(
                    controller: _firstController,
                    label: 'Prénom',
                    icon: Icons.person_rounded,
                    mandatory: true,
                  ),

                  const SizedBox(height: 16),

                  _buildTextField(
                    controller: _lastController,
                    label: 'Nom',
                    icon: Icons.person_outline_rounded,
                    mandatory: true,
                  ),

                  const SizedBox(height: 16),

                  // Section NFC
                  Container(
                    padding: const EdgeInsets.all(12),
                    decoration: BoxDecoration(
                      color: Colors.grey.withOpacity(0.05),
                      borderRadius: BorderRadius.circular(10),
                      border: Border.all(color: Colors.grey.withOpacity(0.2)),
                    ),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          'IDENTIFICATION NFC',
                          style: TextStyle(
                            fontSize: 12,
                            fontWeight: FontWeight.bold,
                            color: Colors.grey.shade600,
                            letterSpacing: 1,
                          ),
                        ),
                        const SizedBox(height: 12),

                        _buildTextField(
                          controller: _macController,
                          label: 'MAC (UID NFC)',
                          icon: Icons.nfc_rounded,
                        ),

                        const SizedBox(height: 12),

                        Row(
                          children: [
                            Container(
                              padding: const EdgeInsets.all(4),
                              decoration: BoxDecoration(
                                color: _hasNfc ? _successColor.withOpacity(0.1) : Colors.grey.withOpacity(0.1),
                                borderRadius: BorderRadius.circular(6),
                                border: Border.all(
                                  color: _hasNfc ? _successColor : Colors.grey,
                                ),
                              ),
                              child: Row(
                                children: [
                                  Checkbox(
                                    value: _hasNfc,
                                    onChanged: (v) => setState(() => _hasNfc = v ?? false),
                                    activeColor: _successColor,
                                  ),
                                  Text(
                                    'NFC détecté',
                                    style: TextStyle(
                                      color: _hasNfc ? _successColor : Colors.grey,
                                      fontWeight: FontWeight.w500,
                                    ),
                                  ),
                                ],
                              ),
                            ),

                            const Spacer(),

                            ElevatedButton.icon(
                              onPressed: _isScanning ? null : _scanNfcAndFill,
                              icon: _isScanning
                                  ? SizedBox(
                                width: 16,
                                height: 16,
                                child: CircularProgressIndicator(
                                  strokeWidth: 2,
                                  color: Colors.white,
                                ),
                              )
                                  : const Icon(Icons.nfc_rounded, size: 18),
                              label: Text(_isScanning ? 'Scan...' : 'Scanner'), // ← TEXTE RACCOURCI
                              style: ElevatedButton.styleFrom(
                                backgroundColor: _primaryColor,
                                foregroundColor: Colors.white,
                                shape: RoundedRectangleBorder(
                                  borderRadius: BorderRadius.circular(8),
                                ),
                                padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 10), // ← PADDING RÉDUIT
                              ),
                            ),

                          ],
                        ),
                      ],
                    ),
                  ),

                  const SizedBox(height: 16),

                  _buildTextField(
                    controller: _qrController,
                    label: 'QR Code (optionnel)',
                    icon: Icons.qr_code_rounded,
                  ),

                  const SizedBox(height: 20),

                  // Bouton de soumission
                  Container(
                    height: 48,
                    decoration: BoxDecoration(
                      borderRadius: BorderRadius.circular(12),
                      boxShadow: [
                        BoxShadow(
                          color: _primaryColor.withOpacity(0.3),
                          blurRadius: 10,
                          offset: const Offset(0, 4),
                        ),
                      ],
                    ),
                    child: ElevatedButton.icon(
                      onPressed: _submit,
                      icon: const Icon(Icons.save_rounded, size: 24),
                      label: const Text(
                        'ENREGISTRER L\'ÉTUDIANT',
                        style: TextStyle(
                          fontSize: 16,
                          fontWeight: FontWeight.bold,
                          letterSpacing: 0.5,
                        ),
                      ),
                      style: ElevatedButton.styleFrom(
                        backgroundColor: _primaryColor,
                        foregroundColor: Colors.white,
                        shape: RoundedRectangleBorder(
                          borderRadius: BorderRadius.circular(12),
                        ),
                      ),
                    ),
                  ),
                ],
              ),
            ),
          ),
        ),
      ),
    );
  }

  Widget _buildTextField({
    required TextEditingController controller,
    required String label,
    required IconData icon,
    bool mandatory = true,
  }) {
    return TextFormField(
      controller: controller,
      decoration: InputDecoration(
        labelText: label,
        prefixIcon: Icon(icon, color: _primaryColor),
        border: OutlineInputBorder(borderRadius: BorderRadius.circular(8)),
      ),
      validator: mandatory
          ? (v) => (v == null || v.trim().isEmpty) ? 'Veuillez remplir ce champ' : null
          : null,
    );
  }

  String _bytesToHexString(dynamic data) {
    if (data == null) return '';
    try {
      if (data is String) return data;
      if (data is Uint8List) return data.map((b) => b.toRadixString(16).padLeft(2, '0')).join(':').toUpperCase();
      if (data is List<int>) return data.map((b) => b.toRadixString(16).padLeft(2, '0')).join(':').toUpperCase();
      if (data is Iterable) return data.map((b) => (b as int).toRadixString(16).padLeft(2, '0')).join(':').toUpperCase();
    } catch (_) {}
    return data.toString();
  }

  String _bytesToUid(dynamic data) {
    final s = _bytesToHexString(data);
    // normalize common separators
    return s.replaceAll('-', ':').replaceAll(' ', ':').replaceAll('::', ':').toUpperCase();
  }
}