<?php

/**
 * Classe PlateauSquadro
 * Cette classe représente le plateau de jeu Squadro, contenant un ensemble de pièces.
 * Elle gère l'initialisation du plateau, la gestion des pièces, et les déplacements possibles.
 */
class PlateauSquadro
{
    const BLANC_V_ALLER = [1, 3, 2, 3, 1];
    const BLANC_V_RETOUR = [3, 1, 2, 1, 3];
    const NOIR_V_ALLER = [3, 1, 2, 1, 3];
    const NOIR_V_RETOUR = [1, 3, 2, 3, 1];

    private $plateau = [];
    private $lignesJouables = [1, 2, 3, 4, 5];
    private $colonnesJouables = [1, 2, 3, 4, 5];

    /**
     * Constructeur de la classe PlateauSquadro.
     * Initialise le plateau avec des cases vides, des cases neutres, et les pièces blanches et noires.
     */
    public function __construct()
    {
        $this->initCasesVides();
        $this->initCasesNeutres();
        $this->initCasesBlanches();
        $this->initCasesNoires();
    }

    /**
     * Initialise toutes les cases du plateau avec des pièces vides.
     */
    private function initCasesVides()
    {
        for ($i = 0; $i < 7; $i++) {
            $this->plateau[$i] = new ArrayPieceSquadro();
            for ($j = 0; $j < 7; $j++) {
                $this->plateau[$i]->add(PieceSquadro::initVide());
            }
        }
    }

    /**
     * Initialise les cases neutres du plateau (les coins).
     */
    private function initCasesNeutres()
    {
        $this->plateau[0][0] = PieceSquadro::initVide();
        $this->plateau[0][6] = PieceSquadro::initVide();
        $this->plateau[6][0] = PieceSquadro::initVide();
        $this->plateau[6][6] = PieceSquadro::initVide();
    }

    /**
     * Initialise les cases blanches du plateau (côté gauche).
     */
    private function initCasesBlanches()
    {
        for ($i = 1; $i <= 5; $i++) {
            $this->plateau[$i][0] = PieceSquadro::initBlancOuest();
        }
    }

    /**
     * Initialise les cases noires du plateau (côté droit).
     */
    private function initCasesNoires()
    {
        for ($j = 1; $j <= 5; $j++) {
            $this->plateau[6][$j] = PieceSquadro::initNoirSud();
        }
    }

    /**
     * Retourne le plateau de jeu.
     * @return array Le plateau de jeu.
     */

    public function getPlateau()
    {
        return $this->plateau;
    }

    /**
     * Récupère une pièce à une position donnée.
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @return PieceSquadro La pièce à la position spécifiée.
     * @throws OutOfBoundsException Si les coordonnées sont invalides.
     */
    public function getPiece($x, $y)
    {
        if ($x < 0 || $x >= 7 || $y < 0 || $y >= 7) {
            throw new OutOfBoundsException("Coordonnées invalides.");
        }
        return $this->plateau[$x][$y];
    }

    /**
     * Définit une pièce à une position donnée.
     * @param PieceSquadro $piece La pièce à définir.
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @throws OutOfBoundsException Si les coordonnées sont invalides.
     */
    public function setPiece($piece, $x, $y)
    {
        if ($x < 0 || $x >= 7 || $y < 0 || $y >= 7) {
            throw new OutOfBoundsException("Coordonnées invalides.");
        }
        $this->plateau[$x][$y] = $piece;
    }

    /**
     * Retourne les lignes jouables du plateau.
     * @return array Les lignes jouables.
     */

    public function getLignesJouables()
    {
        return $this->lignesJouables;
    }

    /**
     * Retourne les colonnes jouables du plateau.
     * @return array Les colonnes jouables.
     */
    public function getColonnesJouables()
    {
        return $this->colonnesJouables;
    }

    /**
     * Retire une ligne jouable du plateau.
     * @param int $index L'index de la ligne à retirer.
     * @throws OutOfBoundsException Si l'index est invalide.
     */
    public function retireLigneJouable($index)
    {
        if (($key = array_search($index, $this->lignesJouables)) !== false) {
            unset($this->lignesJouables[$key]);
        } else {
            throw new OutOfBoundsException("Index de ligne invalide.");
        }
    }

    /**
     * Retire une colonne jouable du plateau.
     * @param int $index L'index de la colonne à retirer.
     * @throws OutOfBoundsException Si l'index est invalide.
     */
    public function retireColonneJouable($index)
    {
        if (($key = array_search($index, $this->colonnesJouables)) !== false) {
            unset($this->colonnesJouables[$key]);
        } else {
            throw new OutOfBoundsException("Index de colonne invalide.");
        }
    }

    /**
     * Retourne les coordonnées de destination d'une pièce en fonction de sa position actuelle.
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @return array Les coordonnées de destination [x, y].
     * @throws OutOfBoundsException Si les coordonnées sont invalides.
     */
    public function getCoordDestination($x, $y)
    {
        if ($x < 0 || $x >= 7 || $y < 0 || $y >= 7) {
            throw new OutOfBoundsException("Coordonnées invalides.");
        }

        $piece = $this->getPiece($x, $y);
        if ($piece->getCouleur() === PieceSquadro::BLANC) {
            $direction = $piece->getDirection();
            $speed = ($direction === PieceSquadro::OUEST) ? self::BLANC_V_ALLER[$y] : self::BLANC_V_RETOUR[$y];
            return [$x, $y + $speed];
        } else {
            $direction = $piece->getDirection();
            $speed = ($direction === PieceSquadro::SUD) ? self::NOIR_V_ALLER[$x] : self::NOIR_V_RETOUR[$x];
            return [$x + $speed, $y];
        }
    }

    /**
     * Retourne la pièce de destination en fonction de la position actuelle.
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @return PieceSquadro La pièce de destination.
     * @throws OutOfBoundsException Si les coordonnées de destination sont invalides.
     */
    public function getDestination($x, $y)
    {
        list($newX, $newY) = $this->getCoordDestination($x, $y);
        if ($newX < 0 || $newX >= 7 || $newY < 0 || $newY >= 7) {
            throw new OutOfBoundsException("Coordonnées de destination invalides.");
        }
        return $this->getPiece($newX, $newY);
    }

    /**
     * Convertit le plateau en JSON.
     * @return string Le plateau sous forme de JSON.
     */
    public function toJson(): string
    {
        $plateauArray = [];
        foreach ($this->plateau as $row) {
            $plateauArray[] = json_decode($row->toJson(), true);
        }
        return json_encode($plateauArray);
    }

    /**
     * Crée une instance de PlateauSquadro à partir d'un JSON.
     * @param string $json Le JSON représentant le plateau.
     * @return self Une instance de PlateauSquadro.
     * @throws InvalidArgumentException Si le JSON est invalide.
     */
    public static function fromJson(string $json): self
    {
        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException("JSON invalide.");
        }
        $plateau = new self();
        foreach ($data as $i => $row) {
            $plateau->plateau[$i] = ArrayPieceSquadro::fromJson(json_encode($row));
        }
        return $plateau;
    }
}
?>
