<?php

/**
 * Classe PieceSquadro
 * 
 * Cette classe représente une pièce du jeu Squadro, avec une couleur et une direction.
 * Elle fournit des méthodes pour manipuler la pièce, comme inverser sa direction ou la convertir en JSON.
 */
class PieceSquadro
{
    const BLANC = 0;
    const NOIR = 1;
    const VIDE = 2;
    const NEUTRE = 3;
    const NORD = 4;
    const EST = 5;
    const SUD = 6;
    const OUEST = 7;

    private $couleur;
    private $direction;

    /**
     * Constructeur de la classe PieceSquadro.
     * 
     * @param int $couleur La couleur de la pièce (BLANC, NOIR, VIDE, NEUTRE).
     * @param int $direction La direction de la pièce (NORD, EST, SUD, OUEST).
     * @throws InvalidArgumentException Si la couleur ou la direction est invalide.
     */
    private function __construct($couleur, $direction)
    {
        if (!in_array($couleur, [self::BLANC, self::NOIR, self::VIDE, self::NEUTRE])) {
            throw new InvalidArgumentException("Couleur invalide.");
        }
        if (!in_array($direction, [self::NORD, self::EST, self::SUD, self::OUEST])) {
            throw new InvalidArgumentException("Direction invalide.");
        }
        $this->couleur = $couleur;
        $this->direction = $direction;
    }

    /**
     * Initialise une pièce vide.
     * 
     * @return PieceSquadro Une pièce vide.
     */
    public static function initVide()
    {
        return new self(self::VIDE, self::NORD);
    }

    /**
     * Initialise une pièce noire orientée vers le nord.
     * 
     * @return PieceSquadro Une pièce noire orientée vers le nord.
     */
    public static function initNoirNord()
    {
        return new self(self::NOIR, self::NORD);
    }

    /**
     * Initialise une pièce noire orientée vers le sud.
     * 
     * @return PieceSquadro Une pièce noire orientée vers le sud.
     */
    public static function initNoirSud()
    {
        return new self(self::NOIR, self::SUD);
    }

    /**
     * Initialise une pièce blanche orientée vers l'est.
     * 
     * @return PieceSquadro Une pièce blanche orientée vers l'est.
     */
    public static function initBlancEst()
    {
        return new self(self::BLANC, self::EST);
    }

    /**
     * Initialise une pièce blanche orientée vers l'ouest.
     * 
     * @return PieceSquadro Une pièce blanche orientée vers l'ouest.
     */
    public static function initBlancOuest()
    {
        return new self(self::BLANC, self::OUEST);
    }

    /**
     * Retourne la couleur de la pièce.
     * 
     * @return int La couleur de la pièce.
     */
    public function getCouleur()
    {
        return $this->couleur;
    }

    /**
     * Retourne la direction de la pièce.
     * 
     * @return int La direction de la pièce.
     */
    public function getDirection()
    {
        return $this->direction;
    }

    /**
     * Inverse la direction de la pièce (NORD <-> SUD).
     * 
     * @throws LogicException Si la direction ne peut pas être inversée.
     */
    public function inverseDirection()
    {
        if ($this->direction === self::NORD || $this->direction === self::SUD) {
            $this->direction = ($this->direction === self::NORD) ? self::SUD : self::NORD;
        } else {
            throw new LogicException("Seules les directions NORD et SUD peuvent être inversées.");
        }
    }

    /**
     * Convertit la pièce en JSON.
     * 
     * @return string La pièce sous forme de JSON.
     */
    public function toJson()
    {
        return json_encode(['couleur' => $this->couleur, 'direction' => $this->direction]);
    }

    /**
     * Crée une pièce à partir d'un JSON.
     * 
     * @param string $json Le JSON représentant la pièce.
     * @return PieceSquadro Une instance de PieceSquadro.
     * @throws InvalidArgumentException Si le JSON est invalide.
     */
    public static function fromJson($json)
    {
        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException("JSON invalide.");
        }
        return new self($data['couleur'], $data['direction']);
    }
}
?>