
<?php

/**

 * Classe ArrayPieceSquadro

 *

 * Cette classe implémente un tableau de pièces Squadro, permettant de manipuler un ensemble de pièces

 * avec des méthodes pour ajouter, supprimer, et accéder aux éléments. Elle implémente les interfaces

 * ArrayAccess et Countable pour un accès similaire à un tableau natif.

 */

class ArrayPieceSquadro implements ArrayAccess, Countable

{

    private $pieces = [];

    /**
     * Vérifie si un offset existe dans le tableau de pièces.
     * @param mixed $offset L'offset à vérifier.
     * @return bool Retourne true si l'offset existe, false sinon.
     */

    public function offsetExists($offset): bool
    {
        return isset($this->pieces[$offset]);
    }

    /**
     * Récupère une pièce à un offset donné.
     * @param mixed $offset L'offset de la pièce à récupérer.
     * @return PieceSquadro La pièce à l'offset spécifié.
     * @throws OutOfBoundsException Si l'offset n'existe pas.
     */

    public function offsetGet($offset) {
        if (!$this->offsetExists($offset)) {
            throw new OutOfBoundsException("Offset invalide : l'index $offset n'existe pas.");
        }
        return $this->pieces[$offset];
    }

    /**
     * Définit une pièce à un offset donné
     * @param mixed $offset L'offset où définir la pièce.
     * @param PieceSquadro $value La pièce à définir.
     * @throws InvalidArgumentException Si la valeur n'est pas une instance de PieceSquadro.
     * @throws OutOfBoundsException Si l'offset est invalide.
     */

    public function offsetSet($offset, $value): void {
        if (!$value instanceof PieceSquadro) {
            throw new InvalidArgumentException("Value must be an instance of PieceSquadro.");
        }
        if (is_null($offset)) {
            $this->pieces[] = $value; // Ajoute à la fin du tableau si l'offset est null
        } else {
            if (!is_int($offset) || $offset < 0) {
                throw new OutOfBoundsException("Offset invalide : l'index doit être un entier positif.");
            }
            $this->pieces[$offset] = $value; // Définit la valeur à l'offset spécifié
        }
    }

    /**
     * Supprime une pièce à un offset donné.
     * @param mixed $offset L'offset de la pièce à supprimer.
     * @throws OutOfBoundsException Si l'offset n'existe pas.
     */

    public function offsetUnset($offset): void {
        if (!$this->offsetExists($offset)) {
            throw new OutOfBoundsException("Offset invalide : l'index $offset n'existe pas.");
        }
        unset($this->pieces[$offset]);
        $this->pieces = array_values($this->pieces); // Réindexe le tableau
    }

    /**
     * Retourne le nombre de pièces dans le tableau.
     * @return int Le nombre de pièces.
     */

    public function count(): int {
        return count($this->pieces);
    }

    /**
     * Ajoute une pièce au tableau.
     * @param PieceSquadro $piece La pièce à ajouter.
     */

    public function add(PieceSquadro $piece): void {
        $this->pieces[] = $piece;
    }

    /**
     * Supprime une pièce à un index donné.
     * @param int $index L'index de la pièce à supprimer.
     * @throws OutOfBoundsException Si l'index n'existe pas.
     */

    public function remove(int $index): void {
        if (!$this->offsetExists($index)) {
            throw new OutOfBoundsException("Index invalide : l'index $index n'existe pas.");
        }
        unset($this->pieces[$index]);
        $this->pieces = array_values($this->pieces); // Réindexe le tableau
    }

    /**
     * Convertit le tableau de pièces en JSON.
     * @return string Le tableau de pièces sous forme de JSON.
     */

    public function toJson(): string {
        return json_encode(array_map(function($piece) {
            return $piece->toJson();
        }, $this->pieces));
    }

    /**
     * Crée une instance de ArrayPieceSquadro à partir d'un JSON.
     * @param string $json Le JSON représentant le tableau de pièces.
     * @return self Une instance de ArrayPieceSquadro.
     * @throws InvalidArgumentException Si le JSON est invalide.
     */

    public static function fromJson(string $json): self {
        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidArgumentException("JSON invalide.");
        }
        $array = new self();
        foreach ($data as $pieceJson) {
            if (!is_array($pieceJson) || !isset($pieceJson['couleur']) || !isset($pieceJson['direction'])) {
                throw new InvalidArgumentException("JSON invalide : chaque élément doit être une pièce valide.");
            }
            $array->add(PieceSquadro::fromJson(json_encode($pieceJson)));
        }
        return $array;
    }
}


?>
