<?php

class ActionSquadro
{
    private $plateau;

    /**
     * Constructeur de la classe ActionSquadro.
     * @param PlateauSquadro $plateau Le plateau de jeu sur lequel les actions seront effectuées.
     */

    public function __construct(PlateauSquadro $plateau)
    {
        $this->plateau = $plateau;
    }

    /**
     * Vérifie si une pièce peut être jouée (c'est-à-dire si elle n'est pas vide).
     * Vérifie si une pièce peut être jouée, c'est-à-dire si elle n'est pas vide.
     * @param int $x La coordonnée X de la pièce sur le plateau.
     * @param int $y La coordonnée Y de la pièce sur le plateau.
     * @return bool Retourne true si la pièce peut être jouée, false sinon.
     */
    public function estJouablePiece($x, $y): bool
    {
        $piece = $this->plateau->getPiece($x, $y);
        return $piece->getCouleur() !== PieceSquadro::VIDE;
    }

    /**
     * Joue une pièce en la déplaçant, en gérant les collisions, les retournements et les sorties.
     * @param int $x La coordonnée X de la pièce à jouer.
     * @param int $y La coordonnée Y de la pièce à jouer.
     * @throws Exception Si la pièce ne peut pas être jouée ou si le mouvement n'est pas en ligne droite.
     */

    public function jouePiece($x, $y): void
    {
        if (!$this->estJouablePiece($x, $y)) {
            throw new Exception("Cette pièce ne peut pas être jouée.");
        }

        $piece = $this->plateau->getPiece($x, $y);
        list($destX, $destY) = $this->plateau->getCoordDestination($x, $y);

        // Vérifie si le mouvement est en ligne droite
        if ($x != $destX && $y != $destY) {
            throw new Exception("Le mouvement doit être en ligne droite.");
        }

        // Gestion des collisions
        $this->gererReculAdverses($x, $y, $destX, $destY);

        // Mise à jour des coordonnées ou inversement de direction
        if ($this->estPointRetournement($destX, $destY, $piece)) {
            $piece->inverseDirection();
        } elseif ($this->estSortieValide($destX, $destY)) {
            $this->sortPiece($x, $y);
        } else {
            $this->plateau->setPiece(PieceSquadro::initVide(), $x, $y);
            $this->plateau->setPiece($piece, $destX, $destY);
        }
    }

    /**
     * Gère le recul des pièces adverses lors d'un mouvement.
     * @param int $startX La coordonnée X de départ.
     * @param int $startY La coordonnée Y de départ.
     * @param int $destX La coordonnée X de destination.
     * @param int $destY La coordonnée Y de destination.
     */

    private function gererReculAdverses($startX, $startY, $destX, $destY): void
    {
        $dx = $destX - $startX;
        $dy = $destY - $startY;
        $steps = max(abs($dx), abs($dy));

        for ($i = 1; $i <= $steps; $i++) {
            $x = $startX + ($dx / $steps) * $i;
            $y = $startY + ($dy / $steps) * $i;

            $adverse = $this->plateau->getPiece($x, $y);
            if ($adverse->getCouleur() !== PieceSquadro::VIDE &&
                $adverse->getCouleur() !== $this->plateau->getPiece($startX, $startY)->getCouleur()) {
                $this->reculerPiece($x, $y);
            }
        }
    }

    /**
     * Recule une pièce adverse au début de son parcours.     *
     * @param int $x La coordonnée X de la pièce à reculer.
     * @param int $y La coordonnée Y de la pièce à reculer.
     */
    public function reculerPiece($x, $y): void
    {
        $piece = $this->plateau->getPiece($x, $y);
        $piece->inverseDirection();
        list($destX, $destY) = $this->plateau->getCoordDestination($x, $y);
        $this->plateau->setPiece(PieceSquadro::initVide(), $x, $y);
        $this->plateau->setPiece($piece, $destX, $destY);
    }

    /**
     * Retire une pièce du plateau lorsqu'elle a terminé son parcours.     *
     * @param int $x La coordonnée X de la pièce à retirer.
     * @param int $y La coordonnée Y de la pièce à retirer.
      */

    public function sortPiece($x, $y): void
    {
        $this->plateau->setPiece(PieceSquadro::initVide(), $x, $y);
    }

    /**
     * Vérifie si une pièce atteint un point de retournement.
     *
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @param PieceSquadro $piece La pièce à vérifier.
     * @return bool Retourne true si la pièce atteint un point de retournement, false sinon.
     */

    private function estPointRetournement($x, $y, $piece): bool
    {
        if ($piece->getCouleur() === PieceSquadro::BLANC && $x === 6) {
            return true;
        }
        if ($piece->getCouleur() === PieceSquadro::NOIR && $y === 0) {
            return true;
        }
        return false;
    }

    /**
     * Vérifie si une pièce sort du plateau.
     * @param int $x La coordonnée X de la pièce.
     * @param int $y La coordonnée Y de la pièce.
     * @return bool Retourne true si la pièce sort du plateau, false sinon.
     */

    private function estSortieValide($x, $y): bool
    {
        return $x < 0 || $x >= 7 || $y < 0 || $y >= 7;
    }

    /**
     * Vérifie si une couleur a remporté la partie.
     * @param int $couleur La couleur à vérifier (PieceSquadro::BLANC ou PieceSquadro::NOIR).
     * @return bool Retourne true si la couleur a remporté la partie, false sinon.
     */
    public function remporteVictoire($couleur): bool
    {
        $piecesSorties = 0;
        for ($i = 0; $i < 7; $i++) {
            for ($j = 0; $j < 7; $j++) {
                $piece = $this->plateau->getPiece($i, $j);
                if ($piece->getCouleur() === $couleur && $this->estSortieValide($i, $j)) {
                    $piecesSorties++;
                }
            }
        }
        return $piecesSorties >= 5; // Une couleur gagne si 5 pièces sont sorties
    }
}
?>
