<?php
// src/DataFixtures/EtablissementFixtures.php

namespace App\DataFixtures;

use App\Entity\Etablissement;
use App\Enum\SecteurEnum;
use App\Enum\EtatEtabEnum;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;

class EtablissementFixtures extends Fixture
{
    public const ETABLISSEMENT_REFERENCE_PREFIX = 'etab_';

    public function load(ObjectManager $manager): void
    {
        $csvFile = __DIR__ . '/data/etablissements.csv';
        if (!file_exists($csvFile)) {
            throw new \Exception("Fichier CSV non trouvé : $csvFile");
        }

        if (($handle = fopen($csvFile, 'r')) !== false) {
            // Lecture de la première ligne (en-tête) pour l’ignorer si nécessaire
            $header = fgetcsv($handle, 1000, ',');

            $batchSize = 20;
            $i = 0;

            while (($data = fgetcsv($handle, 10000, ',')) !== false) {
                $etab = new Etablissement();

                // (0) numero_uai - ignoré
                // (1) appellation_officielle
                $etab->setAppellationOfficielle($data[1]);

                // (2) secteur (Enum)
                // Assurez-vous que la valeur CSV ("Public", "Privé"…) correspond à votre enum
                $etab->setSecteur(SecteurEnum::from($data[2]));

                // (3) adresse
                $etab->setAdresse($data[3]);

                // (4) code_postal
                $etab->setCodePostal((int)$data[4]);

                // (5) commune
                $etab->setCommune($data[5]);

                // (6) EPSG
                $etab->setEPSG($data[6]);

                // (7) latitude
                $etab->setLatitude((float)$data[7]);

                // (8) longitude
                $etab->setLongitude((float)$data[8]);

                // (9) nature
                $etab->setNatureUai((int)$data[9]);

                // (10) naturel_libe
                $etab->setNatureLibe($data[10]);

                // (11) etat_etablissement
                $etab->setEtatEtablissement((int)$data[11]);

                // (12) etat_etablissement_libe (Enum)
                $etab->setEtatEtablissementLibe(EtatEtabEnum::from($data[12]));

                // (13) code_departement
                $etab->setCodeDepartement($data[13]);

                // (14) code_region
                $etab->setCodeRegion((int)$data[14]);

                // (15) code_academie
                $etab->setCodeAcademie((int)$data[15]);

                // (16) code_commune
                $etab->setCodeCommune($data[16]);

                // (17) departement
                $etab->setDepartement($data[17]);

                // (18) region
                $etab->setRegion($data[18]);

                // (19) academie
                $etab->setAcademie($data[19]);

                // (20) type_contrat
                $etab->setTypeContrat(!empty($data[20]) ? (int)$data[20] : null);

                // (21) libelle_type_contrat
                $etab->setLibelleTypeContrat($data[21] ?: null);

                // (22) code_ministere
                $etab->setCodeMinistere((int)$data[22]);

                // (23) libelle_ministere
                $etab->setLibelleMinistere($data[23]);

                // (24) date_ouverture
                $etab->setDateOuverture(new \DateTime($data[24]));

                // Persiste l'entité
                $manager->persist($etab);


                // Gestion du flush/clear par lot
                if (($i % $batchSize) === 0) {
                    $manager->flush();
                    $manager->clear();
                }
                $i++;
            }

            // Fermeture du fichier et flush final
            fclose($handle);
            $manager->flush();
        }
    }
}
