<?php

namespace App\Filter;


use ApiPlatform\Serializer\Filter\FilterInterface;
use DateTime;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\PropertyInfo\Type;

class DateRangeFilter implements FilterInterface
{
    public function getDescription(string $resourceClass): array
    {
        return [
            'granularite' => [
                'property' => null,
                'type' => Type::BUILTIN_TYPE_STRING,
                'required' => true,
                'description' => 'Taille des groupes (jour, mois, annee)'
            ],
            'dateDebut' => [
                'property' => null,
                'type' => Type::BUILTIN_TYPE_STRING,
                'required' => false,
                'description' => 'Date à partir de laquelle compter'
            ],
            'dateFin' => [
                'property' => null,
                'type' => Type::BUILTIN_TYPE_STRING,
                'required' => false,
                'description' => 'Date jusqu\'à  laquelle compter'
            ]
        ];
    }

    public function apply(Request $request, bool $normalization, array $attributes, array &$context): void
    {
        // Granularité
        if ($request->query->has('granularite')) {
            $text = $request->query->get('granularite');

            if (!in_array($text, ['jour', 'mois', 'annee'], true)) {
                throw new BadRequestHttpException("Granularité non reconnue - reçu $text, attendu 'jour' ou 'mois' ou 'annee'");
            }

            $context['granularite'] = $text;
        } else {
            $context['granularite'] = 'mois';
        }

        // Date de début
        if ($request->query->has('dateDebut')) {
            $text = $request->query->get('dateDebut');

            $d = DateTime::createFromFormat('Y-m-d', $text);

            if ($d === false) {
                throw new BadRequestHttpException("Date de début non reconnue - reçu $text, attendu une date au format YYYY-MM-DD");
            }

            $d->setTime(0, 0);

            $context['dateDebut'] = $d;
        } else {
            $context['dateDebut'] = (new DateTime())->setDate(1900, 1, 1)->setTime(0, 0);
        }

        // Date de fin
        if ($request->query->has('dateFin')) {
            $text = $request->query->get('dateFin');

            $d = DateTime::createFromFormat('Y-m-d', $text);

            if ($d === false) {
                throw new BadRequestHttpException("Date de fin non reconnue - reçu $text, attendu une date au format YYYY-MM-DD");
            }

            $d->setTime(23, 59);

            $context['dateFin'] = $d;
        } else {
            $context['dateFin'] = (new DateTime())->setDate(2030, 12, 31)->setTime(23, 59);
        }
    }
}
