<?php

namespace App\Entity;

use ApiPlatform\Action\NotFoundAction;
use ApiPlatform\Metadata\ApiProperty;
use ApiPlatform\Metadata\ApiResource;
use ApiPlatform\Metadata\Get;
use ApiPlatform\Metadata\GetCollection;
use App\State\RegionVentesProvider;
use Symfony\Component\Serializer\Annotation\Groups;

#[ApiResource(operations: [
    new GetCollection(
        uriTemplate: '/ventes/regions',
        description: 'Obtient le nombre de ventes par région',
        provider: RegionVentesProvider::class
    ),
    new Get(
        uriTemplate: '/ventes/regions/{region}',
        controller: NotFoundAction::class,
        output: false,
        read: false
    )
],
    normalizationContext: ['groups' => ['read']],
    paginationEnabled: false)]
class RegionVentes
{
    // Codes INSEE pour les régions métropolitaines et la Corse
    public const AUVERGNE_RHONE_ALPES = 84;
    public const BOURGOGNE_FRANCHE_COMTE = 27;
    public const BRETAGNE = 53;
    public const CENTRE_VAL_DE_LOIRE = 24;
    public const CORSE = 94;
    public const GRAND_EST = 44;
    public const HAUTS_DE_FRANCE = 32;
    public const ILE_DE_FRANCE = 11;
    public const NORMANDIE = 28;
    public const NOUVELLE_AQUITAINE = 75;
    public const OCCITANIE = 76;
    public const PAYS_DE_LA_LOIRE = 52;
    public const PROVENCE_ALPES_COTE_AZUR = 93;

    // Zéro pour les autres régions
    public const OUTRE_MER = 0;

    private int $region = -1;
    private int $count = 0;

    #[Groups('read')]
    #[ApiProperty(identifier: true)]
    public function getRegion(): int
    {
        return $this->region;
    }

    public function setRegion(int $region): self
    {
        $this->region = $region;
        return $this;
    }

    #[Groups('read')]
    public function getRegionName(): string
    {
        switch ($this->region) {
            case RegionVentes::AUVERGNE_RHONE_ALPES:
                return "Auvergne-Rhône-Alpes";
            case RegionVentes::BOURGOGNE_FRANCHE_COMTE:
                return "Bourgogne-Franche-Comté";
            case RegionVentes::BRETAGNE:
                return "Bretagne";
            case RegionVentes::CENTRE_VAL_DE_LOIRE:
                return "Centre-Val de Loire";
            case RegionVentes::CORSE:
                return "Corse";
            case RegionVentes::GRAND_EST:
                return "Grand-Est";
            case RegionVentes::HAUTS_DE_FRANCE:
                return "Hauts-de-France";
            case RegionVentes::ILE_DE_FRANCE:
                return "Île-de-France";
            case RegionVentes::NORMANDIE:
                return "Normandie";
            case RegionVentes::NOUVELLE_AQUITAINE:
                return "Nouvelle-Aquitaine";
            case RegionVentes::OCCITANIE:
                return "Occitanie";
            case RegionVentes::PAYS_DE_LA_LOIRE:
                return "Pays de la Loire";
            case RegionVentes::PROVENCE_ALPES_COTE_AZUR:
                return "Provence-Alpes-Côte d'Azur";
            case 0:
                return "Outre-mer";
            default:
                return "Inconnue";
        }
    }

    #[Groups('read')]
    public function getCount(): int
    {
        return $this->count;
    }

    public function setCount(int $count): self
    {
        $this->count = $count;
        return $this;
    }

    public function addCount(int $count): void
    {
        $this->count += $count;
    }
}
