<?php

namespace App\DataFixtures;

use App\Entity\Vente;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;
use DateTime;

class VentesFixtures extends Fixture
{
    public function load(ObjectManager $manager): void
    {
        $output_dir = dirname(__DIR__) . '/../data';

        $files = [
            "2017s2" => "https://www.data.gouv.fr/fr/datasets/r/7161c9f2-3d91-4caf-afa2-cfe535807f04",
            "2018"   => "https://www.data.gouv.fr/fr/datasets/r/1be77ca5-dc1b-4e50-af2b-0240147e0346",
            "2019"   => "https://www.data.gouv.fr/fr/datasets/r/3004168d-bec4-44d9-a781-ef16f41856a2",
            "2020"   => "https://www.data.gouv.fr/fr/datasets/r/90a98de0-f562-4328-aa16-fe0dd1dca60f",
            "2021"   => "https://www.data.gouv.fr/fr/datasets/r/817204ac-2202-4b4a-98e7-4184d154d98c",
            "2022s1" => "https://www.data.gouv.fr/fr/datasets/r/87038926-fb31-4959-b2ae-7a24321c599a",
        ];

        ini_set("memory_limit", "2G");

        $start = microtime(true);

        foreach ($files as $name => $url) {
            $outName = "$output_dir/$name.txt";

            if (file_exists($outName)) {
                print("Le fichier $name.txt existe déjà\n");
                continue;
            }

            if (!$this->downloadFile($url, $outName)) {
                print("Impossible de télécharger le fichier pour $name\n");
                unlink($outName);
            }
        }

        $mid = microtime(true);
        print("Récupération des fichiers: " . gmdate("H:i:s", $mid-$start) . "." . (($mid-$start)*1000)%1000 . PHP_EOL);

        $total = 0;

        foreach ($files as $name => $_) {
            $outName = "$output_dir/$name.txt";

            $total += $this->loadSingleFile($outName, $manager);
        }
        $end = microtime(true);

        print("Chargement des fichiers: " . gmdate("H:i:s", $end-$mid) . PHP_EOL);
        print("Temps total: " . gmdate("H:i:s", $end-$start) . PHP_EOL);
        print("$total entités insérées\n");
    }

    private function downloadFile(String $url, String $out): bool
    {
        print("Téléchargement de $url vers $out\n");
        return file_put_contents($out, file_get_contents($url));
    }

    private function loadSingleFile(String $file, ObjectManager $manager):int
    {
        // Limites, pour tester.
        $maxEntities = 50_000;

        $fp = fopen($file, 'r');

        if ($fp === false) {
            print("Impossible d'ouvrir le fichier de ventes\n");
            return 0;
        }

        print("Fichier $file ouvert\n");

        // Ignorer ligne d'entêtes
        fgets($fp);

        $i = 0;
        $added = 0;

        while (($row = fgetcsv($fp, 0, '|'))) {

            if ($row[8] == "" || $row[10] == "" || $row[18] == "" || $row[35] == "" || $row[38] == "") {
                continue;
            }

            $vente = new Vente();

            $vente->setDate(DateTime::createFromFormat("d/m/Y", $row[8]));
            $vente->setValeur((float)$row[10]);
            $vente->setDepartement((int)$row[18]);
            $vente->setCodeType((int)$row[35]);
            $vente->setSurface((int)$row[38]);

            $manager->persist($vente);
            $added++;
            if ($added % 1000 == 0) {
                $memory = round(memory_get_usage() / (1024*1024), 2);
                print("\x0D$added entités insérées (Mémoire: $memory MB)");
            }

            if ($added % 50000 == 0) {
                print(" - flush...");
                $manager->flush();
                print("\033[2K");
                $manager->clear();
            }

            if ($added >= $maxEntities) {
                break;
            }
        }

        $manager->flush();
        $manager->clear();

        print("\x0D[FIN] $added entités ajoutées\n");

        return $added;
    }
}
