const chai = require('chai');
const chaiHttp = require('chai-http');
const app = require('../app');
const models = require("../schemas/models"); // Import your Express app
const expect = chai.expect;
const fs = require('fs');

chai.use(chaiHttp);

describe('Authentication Routes', () => {
    describe('GET /login', () => {
        it('should render the login page', (done) => {
            chai.request(app)
                .get('/login')
                .end((err, res) => {
                    expect(res).to.have.status(200);
                    expect(res.text).to.include('Connexion');
                    done();
                });
        });
    });

    describe('POST /login', () => {
        it('should authenticate a user with valid credentials', (done) => {
            chai.request(app)
                .post('/login')
                .send({ username: 'hajar', password: 'hajar' })
                .end((err, res) => {
                    if (err) {
                        console.error(err);
                    }
                    expect(res).to.have.status(200);
                    done();
                });
        });
        it('should reject invalid credentials', (done) => {
            chai.request(app)
                .post('/login')
                .send({ username: 'invalidUser', password: 'invalidPassword' })
                .end((err, res) => {
                    expect(res).to.have.status(400);
                    expect(res.text).to.include('Invalid username or password');
                    done();
                });
        });
    });

});

describe('POST /register', () => {
    it('should create a new user agent', async () => {
        const agent = chai.request.agent(app);

        const userData = {
            username: 'testuser',
            password: 'testpassword',
            role: 'agent'
        };

        const res = await agent
            .post('/register')
            .send(userData);

        expect(res).to.redirect;
        expect(res).to.have.status(200);

        await new Promise(resolve => setTimeout(resolve, 500));
        await User.deleteOne({ username: userData.username });
    });

    it('should create a new user consultant', async () => {
        const agent = chai.request.agent(app);

        const userData = {
            username: 'testuser',
            password: 'testpassword',
            role: 'agent'
        };

        const res = await agent
            .post('/register')
            .send(userData);

        expect(res).to.redirect;
        expect(res).to.have.status(200);
        await new Promise(resolve => setTimeout(resolve, 500));
        await User.deleteOne({ username: userData.username });
    });

    it('should return an error for an existing user', async () => {
        const agent = chai.request.agent(app);

        const existingUsername = 'firdaous';

        const userData = {
            username: existingUsername,
            password: 'testpassword',
            role: 'agent'
        };

        const res = await agent
            .post('/register')
            .send(userData);

        expect(res.text).to.include('This user already exists.');
        expect(res).to.have.status(200);

    });
});

describe('GET /logout', () => {
    it('should log out an authenticated user', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'rahmouni', password: 'rahmouni' });

        const res = await agent
            .get('/logout');

        expect(res).to.redirect;
        expect(res).to.have.status(200);
    });

    it('should redirect to login page for a non-authenticated user', async () => {
        const agent = chai.request.agent(app);

        const res = await agent
            .get('/logout');

        expect(res).to.redirect;
        expect(res.text).to.include('Connexion');
    });
});

// ------------------------ Announcements --------------------------------------
const Announcement = models.Announcement;
const User = models.User;

describe('GET /announcements', () => {
    it('should render announcements for non-authenticated user', async () => {
        const res = await chai.request(app)
            .get('/announcements');

        expect(res).to.have.status(200);
    });

    it('should render announcements for authenticated agent', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'elhalafi', password: 'elhalafi' });

        const res = await agent
            .get('/announcements');

        expect(res).to.have.status(200);

        await agent
            .get('/logout')
    });

    it('should render announcements for authenticated consultant', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'firdaous', password: 'firdaous' });

        const res = await agent
            .get('/announcements');

        expect(res).to.have.status(200);

        await agent
            .get('/logout')
    });
});

describe('GET /add-announcement', () => {
    it('should render add announcement for authenticated agent', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'elhalafi', password: 'elhalafi' });

        const res = await agent
            .get('/announcements/add-announcement');

        expect(res).to.have.status(200);

        await agent
            .get('/logout')
    });

    it('should render add announcement for authenticated consultant', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'hajar', password: 'hajar' });

        const res = await agent
            .get('/announcements/add-announcement');

        expect(res.text).to.include('Vous n\'êtes pas autorisé à effectuer cette action. Créez un compte agent pour publier des annonces.');

        await agent
            .get('/logout')
    });
});

describe('GET /announcements/:id', () => {
    it('should render announcement details for a consultant', async () => {
        const agent = chai.request.agent(app);
        /*
        const Agent = new User({
            username: 'ff',
            password: 'ff',
            isAgent: true
        });

        await Agent.save();

        const consultant = new User({
            username: 'aa',
            password: 'aa',
            isAgent: false
        });

        await consultant.save();
        */
        await agent
            .post('/login')
            .send({ username: "firdaous", password: "firdaous" });

        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Loué',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: "elhalafi"
        });

        await announcement.save();

        const res = await agent
            .get(`/announcements/${announcement._id}`)
            .send();

        expect(res.text).to.include('Exemple de titre');
        expect(res).to.have.status(200);

        await Announcement.deleteOne({ _id: announcement._id });
        //await User.deleteOne({ username: consultant.username });
        await agent.get('/logout');
    });

    it('should render announcement details for the agent who posted it', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: "elhalafi", password: "elhalafi" });

        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Loué',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: "elhalafi"
        });

        await announcement.save();

        const res = await agent
            .get(`/announcements/${announcement._id}`)
            .send();

        expect(res.text).to.include('Exemple de titre');
        expect(res).to.have.status(200);

        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });

    it('should render announcement details for a different agent', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: "elhalafi", password: "elhalafi" });

        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Loué',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: "rahmouni"
        });

        await announcement.save();

        const res = await agent
            .get(`/announcements/${announcement._id}`)
            .send();

        expect(res.text).to.include('User non autorisée.');
        expect(res).to.have.status(403);

        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });

    it('advertisement not published', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: "firdaous", password: "firdaous" });

        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Non publiée',
            propertyStatus: 'Loué',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: "elhalafi"
        });

        await announcement.save();

        const res = await agent
            .get(`/announcements/${announcement._id}`)
            .send();

        expect(res.text).to.include('Cette annonce est introuvable ou non disponible.');
        expect(res).to.have.status(404);

        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });

});

describe('POST /announcements/:id/ask', () => {

    it('ask a question if the user is a consultant', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'elhalafi'
        });

        await announcement.save();
        const agent = chai.request.agent(app);
        await agent
            .post('/login')
            .send({username: 'firdaous', password: 'firdaous'});

        const res = await agent
            .post(`/announcements/${announcement._id}/ask`)
            .send({question: 'Exemple de question'});

        expect(res).to.have.status(200);
        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });

    it('should render announcement details for a consultant', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Loué',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });

        await announcement.save();

        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({username: 'hajar', password: 'hajar'});

        const res = await agent
            .post(`/announcements/${announcement._id}/ask`)
            .send({question: 'Exemple de question'});

        expect(res.text).to.include('Vous ne pouvez poser une question que sur les annonces non disponibles.');

        expect(res).to.have.status(403);
        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });
});

describe('POST /announcements/:id/question/:questionId/answer', () => {
    it('should post an answer to a question for an authenticated agent', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });
        await announcement.save();

        const question = {
            user: 'hajar',
            question: 'Exemple de question'
        };

        announcement.questions.push(question);
        await announcement.save();

        const agent = chai.request.agent(app);
        await agent
            .post('/login')
            .send({ username: 'rahmouni', password: 'rahmouni' });

        const questionId = announcement.questions[0]._id;
        const res = await agent
            .post(`/announcements/${announcement._id}/question/${questionId}/answer`)
            .send({ answer: 'Exemple de réponse' });

        expect(res).to.have.status(200);
        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });

    it('Unauthorized user', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });
        await announcement.save();

        const question = {
            user: 'hajar',
            question: 'Exemple de question'
        };

        announcement.questions.push(question);
        await announcement.save();

        const agent = chai.request.agent(app);
        await agent
            .post('/login')
            .send({ username: 'elhalafi', password: 'elhalafi' });

        const questionId = announcement.questions[0]._id;
        const res = await agent
            .post(`/announcements/${announcement._id}/question/${questionId}/answer`)
            .send({ answer: 'Exemple de réponse' });

        expect(res.text).to.include('User non autorisée.');
        expect(res).to.have.status(403);
        await Announcement.deleteOne({ _id: announcement._id });
        await agent.get('/logout');
    });
});

describe('POST /add-announcement', () => {
    it('should add a new announcement for an authenticated agent', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'rahmouni', password: 'rahmouni' });

        const announcementData = {
            title: 'Titre de l\'annonce',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: '2023-10-15',
        };

        const filePath = 'public/uploads/1696762934167-images.jpeg';
        const stats = fs.statSync(filePath);
        const fileSizeInBytes = stats.size;

        const res = await agent
            .post('/announcements/add-announcement')
            .field('title', announcementData.title)
            .field('propertyType', announcementData.propertyType)
            .field('publicationStatus', announcementData.publicationStatus)
            .field('propertyStatus', announcementData.propertyStatus)
            .field('description', announcementData.description)
            .field('price', announcementData.price)
            .field('availabilityDate', announcementData.availabilityDate)
            .attach('photos[]', filePath, {
                filename: 'image.jpg',
                contentType: 'image/jpeg',
                knownLength: fileSizeInBytes
            });

        expect(res).to.have.status(200);

        await new Promise(resolve => setTimeout(resolve, 1500));
        const announcementId = res.body._id;
        await Announcement.deleteOne({ _id: announcementId });
        const uploadedFilePath = `/public/uploads/${res.body.photos[0]}`;
        if (fs.existsSync(uploadedFilePath)) {
            fs.unlinkSync(uploadedFilePath);
        }
        await agent.get('/logout');
    });

    it('should add a new announcement for an authenticated consultant', async () => {
        const agent = chai.request.agent(app);

        await agent
            .post('/login')
            .send({ username: 'hajar', password: 'hajar' });

        const announcementData = {
            title: 'Titre de l\'annonce',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Descriptqijgjejgion de l\'annonce',
            price: 100000,
            availabilityDate: '2023-10-15',
        };

        const filePath = 'public/uploads/1696762934167-images.jpeg';
        const stats = fs.statSync(filePath);
        const fileSizeInBytes = stats.size;

        const res = await agent
            .post('/announcements/add-announcement')
            .field('title', announcementData.title)
            .field('propertyType', announcementData.propertyType)
            .field('publicationStatus', announcementData.publicationStatus)
            .field('propertyStatus', announcementData.propertyStatus)
            .field('description', announcementData.description)
            .field('price', announcementData.price)
            .field('availabilityDate', announcementData.availabilityDate)
            .attach('photos', filePath, {
                filename: 'image.jpg',
                contentType: 'image/jpeg',
                knownLength: fileSizeInBytes
            });

        expect(res.text).to.include('Vous n\'êtes pas autorisé à effectuer cette action.');
        expect(res).to.have.status(403);

        const announcementId = res.body._id;
        await Announcement.deleteOne({ _id: announcementId });
        await agent.get('/logout');
    });
});

describe('GET /announcements/delete/:id', () => {
    it('should delete an announcement for an authenticated agent who s not the owner of the announcement', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });

        await announcement.save();

        const agent = chai.request.agent(app);
        await agent
            .post('/login')
            .send({ username: 'elhalafi', password: 'elhalafi' });


        const res = await agent
            .get(`/announcements/delete/${announcement._id}`)

        expect(res).to.have.status(403);
        await Announcement.deleteOne({ _id: announcement._id });
        //const deletedAnnouncement = await Announcement.findById(announcement._id);
        //expect(deletedAnnouncement).to.be.null;
    });

    it('should delete an announcement for an authenticated agent', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 100000,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });

        await announcement.save();

        const agent = chai.request.agent(app);
        await agent
            .post('/login')
            .send({ username: 'rahmouni', password: 'rahmouni' });


        const res = await agent
            .get(`/announcements/delete/${announcement._id}`)

        expect(res).to.have.status(200);
        const deletedAnnouncement = await Announcement.findById(announcement._id);
        expect(deletedAnnouncement).to.be.null;
    });

    it('should not delete an announcement for an authenticated consultant', async () => {
        const announcement = new Announcement({
            title: 'Exemple de titre- 2',
            propertyType: 'À la vente',
            publicationStatus: 'Publiée',
            propertyStatus: 'Disponible',
            description: 'Description de l\'annonce',
            price: 200,
            availabilityDate: new Date('2023-10-15'),
            userName: 'rahmouni'
        });

        await announcement.save();

        const consultant = chai.request.agent(app);
        await consultant
            .post('/login')
            .send({ username: 'hajar', password: 'hajar' });


        const res = await consultant
            .get(`/announcements/delete/${announcement._id}`)

        expect(res).to.have.status(403);
        const deletedAnnouncement = await Announcement.findById(announcement._id);
        expect(deletedAnnouncement).not.to.be.null;
    });

    describe('POST /announcements/:id/update', () => {
        it('should allow an agent to update its announcement', async function () {

            const announcement = new Announcement({
                title: 'Exemple de titre',
                propertyType: 'À la vente',
                publicationStatus: 'Publiée',
                propertyStatus: 'Disponible',
                description: 'Description de l\'annonce',
                price: 10000,
                availabilityDate: new Date('2023-10-15'),
                userName: 'elhalafi'
            });

            await announcement.save();

            const agentAgent = chai.request.agent(app);
            await agentAgent
                .post('/login')
                .send({ username: 'elhalafi', password: 'elhalafi' });

            const updatedAnnouncementData = {
                title: 'updated title',
                description: 'Description de l\'annonce updated',
            };
            const filePath = 'public/uploads/1696765095124-image.jpg';
            const stats = fs.statSync(filePath);
            const fileSizeInBytes = stats.size;
            const res = await agentAgent
                .post(`/announcements/update/${announcement._id}`)
                .field('title', updatedAnnouncementData.title)
                .field('description', updatedAnnouncementData.description)
                .attach('photos[]', filePath, {
                    filename: '1696765095124-image.jpg',
                    contentType: 'image/jpeg',
                    knownLength: fileSizeInBytes
                });

            expect(res).to.have.status(200);
            const updatedAnnouncement = await Announcement.findById(announcement._id);
            expect(updatedAnnouncement.title).to.equal(updatedAnnouncementData.title);
            expect(updatedAnnouncement.description).to.equal(updatedAnnouncementData.description);
        });

        it('should allow an agent who is not the owner of the ad to update it', async function () {

            const announcement = new Announcement({
                title: 'Exemple de titre',
                propertyType: 'À la vente',
                publicationStatus: 'Publiée',
                propertyStatus: 'Disponible',
                description: 'Description de l\'annonce',
                price: 10000,
                availabilityDate: new Date('2023-10-15'),
                userName: 'elhalafi'
            });

            await announcement.save();

            const agentAgent = chai.request.agent(app);
            await agentAgent
                .post('/login')
                .send({ username: 'rahmouni', password: 'rahmouni' });

            const updatedAnnouncementData = {
                title: 'updated title',
                description: 'Description de l\'annonce updated',
            };
            const filePath = 'public/uploads/1696765095124-image.jpg';
            const stats = fs.statSync(filePath);
            const fileSizeInBytes = stats.size;
            const res = await agentAgent
                .post(`/announcements/update/${announcement._id}`)
                .field('title', updatedAnnouncementData.title)
                .field('description', updatedAnnouncementData.description)
                .attach('photos[]', filePath, {
                    filename: '1696765095124-image.jpg',
                    contentType: 'image/jpeg',
                    knownLength: fileSizeInBytes
                });

            expect(res).to.have.status(403);
            const updatedAnnouncement = await Announcement.findById(announcement._id);
            expect(updatedAnnouncement.title).to.equal(updatedAnnouncementData.title);
            expect(updatedAnnouncement.description).to.equal(updatedAnnouncementData.description);
        });

        it('should not allow a consultant to update an announcement', async function () {

            const announcement = new Announcement({
                title: 'T1 meublé',
                propertyType: 'À la vente',
                publicationStatus: 'Publiée',
                propertyStatus: 'Disponible',
                description: 'Description de l\'annonce',
                price: 10000,
                availabilityDate: new Date('2023-10-15'),
                userName: 'elhalafi'
            });

            await announcement.save();

            const consultantAgent = chai.request.agent(app);
            await consultantAgent
                .post('/login')
                .send({ username: 'firdaous', password: 'firdaous' });

            const updatedAnnouncementData = {
                title: 'updated title',
                description: 'Description de l\'annonce updated',
            };
            const filePath = 'public/uploads/1696765095124-image.jpg';
            const stats = fs.statSync(filePath);
            const fileSizeInBytes = stats.size;
            const res = await consultantAgent
                .post(`/announcements/update/${announcement._id}`)
                .field('title', updatedAnnouncementData.title)
                .field('description', updatedAnnouncementData.description)
                .attach('photos[]', filePath, {
                    filename: '1696765095124-image.jpg',
                    contentType: 'image/jpeg',
                    knownLength: fileSizeInBytes
                });

            expect(res).to.have.status(403);

            // ON s'assure que l'annonce n'a pas été mise à jour
            const updatedAnnouncement = await Announcement.findById(announcement._id);
            expect(updatedAnnouncement.title).to.equal(announcement.title);
            expect(updatedAnnouncement.description).to.equal(announcement.description);
        });

    });

    describe('DELETE /delete-photo/:announcementId/:filename', () => {
        it('should allow an agent to delete a photo from an announcement', async () => {
            const agentAgent = chai.request.agent(app);
            await agentAgent
                .post('/login')
                .send({ username: 'elhalafi', password: 'elhalafi' });

            const announcement = new Announcement({
                title: 'Exemple de titre',
                propertyType: 'À la vente',
                publicationStatus: 'Publiée',
                propertyStatus: 'Disponible',
                description: 'Description de l\'annonce',
                price: 10000,
                availabilityDate: new Date('2023-10-15'),
                userName: 'elhalafi',
                photos: ['1696765095124-image.jpg', '1696770072099-image.jpg']
            });

            await announcement.save();

            const announcementId = announcement._id;
            const filenameToDelete = '1696765095124-image.jpg';
            console.log(announcement);
            const res = await agentAgent
                .delete(`/announcements/delete-photo/${announcementId}/${filenameToDelete}`);

            expect(res).to.have.status(200);
            expect(res.body.message).to.equal('Photo supprimée avec succès');

            const updatedAnnouncement = await Announcement.findById(announcementId);
            expect(updatedAnnouncement.photos).to.not.include(filenameToDelete);
        });
        it('should allow an agent to delete a photo from an announcement that is not its annoucement', async () => {
            const agentAgent = chai.request.agent(app);
            await agentAgent
                .post('/login')
                .send({ username: 'hajar', password: 'hajar' });

            const announcement = new Announcement({
                title: 'Exemple de titre',
                propertyType: 'À la vente',
                publicationStatus: 'Publiée',
                propertyStatus: 'Disponible',
                description: 'Description de l\'annonce',
                price: 10000,
                availabilityDate: new Date('2023-10-15'),
                userName: 'elhalafi',
                photos: ['1696765095124-image.jpg', '1696770072099-image.jpg']
            });

            await announcement.save();

            const announcementId = announcement._id;
            const filenameToDelete = '1696765095124-image.jpg';
            console.log(announcement);
            const res = await agentAgent
                .delete(`/announcements/delete-photo/${announcementId}/${filenameToDelete}`);

            expect(res).to.have.status(403);
            expect(res.body.message).to.equal('Vous n\'êtes pas autorisé à effectuer cette action.');

            const updatedAnnouncement = await Announcement.findById(announcementId);
            expect(updatedAnnouncement.photos).to.include(filenameToDelete);
        });
    });

});


