const express = require('express');
const router = express.Router();
const models = require('../schemas/models');
const routes = require('./index');
const upload = require('../multer-config');
const fs = require('fs');
const path = require('path');


const Announcement = models.Announcement;
const isAuthenticated = routes.isAuthenticated;
const isAgent = routes.isAgent;
const isAgentAddAnnouncements = routes.isAgentAddAnnouncements;
const isConsultant = routes.isConsultant;

// Route to retrieve all ads
router.get('/', async (req, res) => {
    try {
        const user = req.user;
        let announcements;
        if (user && user.isAgent){
            announcements = await Announcement.find({ userName: user.username });
        }
        else {
            announcements = await Announcement.find({ publicationStatus: 'Publiée' });
        }
        let isAuthenticated = false;
        if (req.isAuthenticated()) {
            isAuthenticated = true;
        }
        res.render('announcements', { announcements: announcements, isAuthenticated: isAuthenticated, user: req.user });
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

// Route to display the add ad page
router.get('/add-announcement', isAuthenticated, isAgentAddAnnouncements, (req, res) => {
    const errors = req.flash('error');
    if (errors.length > 0) {
        return res.render('register',  { error: errors[0] });
    }
    res.render('add_announcement',{ user : req.user });
});

// Ad details
router.get('/:id', async (req, res) => {
    try {
        const announcementId = req.params.id;
        const user = req.user;
        const announcementDetails = await Announcement.findById(announcementId);
        if ((user && user.username !==  announcementDetails.userName || !user) && announcementDetails.publicationStatus !== 'Publiée') {
            return res.status(404).json({ message: 'Cette annonce est introuvable ou non disponible.' });
        }
        if (user && user.username !==  announcementDetails.userName && user.isAgent) {
            return res.status(403).json({ message: 'User non autorisée.' });
        }
        res.render('announcement_details', { announcement: announcementDetails, user: req.user });
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

router.post('/:id/ask', isAuthenticated, isConsultant, async (req, res) => {
    const announcementId = req.params.id;
    const { question } = req.body;
    const user = req.user;

    try {
        const announcement = await Announcement.findById(announcementId);

        if (!announcement) {
            return res.status(404).json({ message: 'Annonce non trouvée' });
        }

        if (announcement.propertyStatus !== 'Disponible') {
            return res.status(403).json({ message: 'Vous ne pouvez poser une question que sur les annonces non disponibles.' });
        }

        announcement.questions.push({
            user: user.username,
            question
        });

        await announcement.save();

        res.redirect(`/announcements/${announcementId}`);
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

router.post('/:id/question/:questionId/answer', isAuthenticated, isAgent, async (req, res) => {
    const announcementId = req.params.id;
    const questionId = req.params.questionId;
    const { answer } = req.body;
    const user = req.user;

    try {
        const announcement = await Announcement.findById(announcementId);

        if (!announcement) {
            return res.status(404).json({ message: 'Annonce non trouvée' });
        }

        if (user.username !== announcement.userName) {
            return res.status(403).json({ message: 'User non autorisée.' });
        }

        const question = announcement.questions.id(questionId);

        if (!question) {
            return res.status(404).json({ message: 'Question non trouvée' });
        }

        question.answers.push({ user: user.username, answer });
        await announcement.save();

        res.redirect(`/announcements/${announcementId}`);
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

router.post('/add-announcement', isAuthenticated, isAgent, upload.array('photos[]', ), async (req, res) => {
    const { title, propertyType, publicationStatus, propertyStatus, description, price, availabilityDate} = req.body;
    const photos = [];

    req.files.forEach(file => {
        photos.push(file.filename);
    });

    const user = req.user;

    const newAnnouncement = new Announcement({
        title,
        propertyType,
        publicationStatus,
        propertyStatus,
        description,
        price,
        availabilityDate,
        photos,
        userName: user.username
    });

    try {
        await newAnnouncement.save();
        //res.status(200).json(newAnnouncement);
        res.redirect('/announcements');
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

// Route to display form to update an ad
router.get('/update/:id', isAuthenticated, isAgent, async (req, res) => {
    try {
        const user = req.user;
        const announcementId = req.params.id;

        const announcementDetails = await Announcement.findById(announcementId);
        if (!announcementDetails) {
            return res.status(404).json({ message: 'Announcement not found' });
        }

        if (user.username !== announcementDetails.userName) {
            return res.status(403).json({ message: 'User non autorisée' });
        }

        res.render('update_announcement', { announcement: announcementDetails, user: req.user });
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

// Route to handle update of an ad
router.post('/update/:id', isAuthenticated, isAgent, upload.array('photos[]'), async (req, res) => {
    const announcementId = req.params.id;
    const user = req.user;
    const { title, propertyType, publicationStatus, propertyStatus, description, price, availabilityDate } = req.body;
    let photos = [];

    const oldPhotos = req.body.oldPhotos || [];

    req.files.forEach(file => {
        photos.push(file.filename);
    });

    photos = photos.concat(oldPhotos);

    try {
        const updatedAnnouncement = await Announcement.findByIdAndUpdate(
            announcementId,
            {
                title,
                propertyType,
                publicationStatus,
                propertyStatus,
                description,
                price,
                availabilityDate,
                photos, // we combine old and new photos
            },
            { new: true }
        );

        if (!updatedAnnouncement) {
            return res.status(404).json({ message: 'annonce non trouvée' });
        }

        if (user.username !== updatedAnnouncement.userName) {
            return res.status(403).json({ message: 'User non autorisée' });
        }
        res.redirect('/announcements');
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});

// route to delete an ad
router.get('/delete/:id', isAuthenticated, isAgent, async (req, res) => {
    try {
        const announcementId = req.params.id;
        const user = req.user;

        const deletedAnnouncement = await Announcement.findByIdAndRemove(announcementId);
        if (!deletedAnnouncement) {
            console.log('404')
            return res.status(404).json({ message: 'annonce non trouvée' });
        }
        if (user.username !== deletedAnnouncement.userName) {
            console.log('403')
            return res.status(403).json({ message: 'User non autorisé' });
        }

        deletedAnnouncement.photos.forEach((photo) => {
            const filePath = path.join('public', 'uploads', photo);
            console.log(fs.existsSync(filePath));
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }
        });

        res.redirect('/announcements');
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
});


// Route to delete a photo
router.delete('/delete-photo/:announcementId/:filename', isAuthenticated, isAgent, async (req, res) => {
    try {
        console.log('here ')
        const filename = req.params.filename;
        const announcementId = req.params.announcementId;
        const user = req.user;
        const announcement = await Announcement.findById(announcementId);

        if (!announcement) {
            console.log('Annonce non trouvée')
            return res.status(404).json({ message: 'Annonce non trouvée' });
        }

        if (!announcement.photos.includes(filename)) {
            console.log('Photo non trouvée')
            return res.status(404).json({ message: 'Photo non trouvée' });
        }

        if (user.username !== announcement.userName) {
            return res.status(403).json({ message: 'User non autorisée' });
        }

        const filePath = path.join('public', 'uploads', filename);

        if (fs.existsSync(filePath)) {
            fs.unlinkSync(filePath);

            let updatedPhotos = [...announcement.photos];

            updatedPhotos = updatedPhotos.filter((photo) => photo !== filename);
            announcement.photos = updatedPhotos;
            await announcement.save();

            return res.status(200).json({ message: 'Photo supprimée avec succès' });
        } else {
            console.log('Fichier non trouvé')
            return res.status(404).json({ message: 'Fichier introuvable' });
        }
    } catch (error) {
        console.error('Une erreur s\'est produite :', error);
        return res.status(500).json({ message: 'Erreur interne du serveur' });
    }
});

module.exports = router;