import React, { useState, useEffect } from 'react';
import { Text, View, StyleSheet, Button, TextInput } from 'react-native';
import { BarCodeScanner } from 'expo-barcode-scanner';
import Constants from 'expo-constants';
import { ImageBackground } from 'react-native';
import Styles from '../styles/Style';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import db from '../database/db';

export default function App() {
  const [hasPermission, setHasPermission] = useState(false);
  const [scanned, setScanned] = useState(false);
  const [manualEntry, setManualEntry] = useState(false);
  const [barcodeData, setBarcodeData] = useState('');
  const [itemName, setItemName] = useState('');
  const [itemPrice, setItemPrice] = useState('');
  const navigation = useNavigation();

  useEffect(() => {
    const getBarCodeScannerPermissions = async () => {
      const { status } = await BarCodeScanner.requestPermissionsAsync();
      const isPermissionGranted = status === 'granted';
      setHasPermission(isPermissionGranted);

      if (!isPermissionGranted) {
        setManualEntry(true);
      }
    };
    getBarCodeScannerPermissions();
  }, []);

  const extractDataFromBarcode = (data) => {
    const [name, price] = data.split('-');
    console.log('extract : ', name, ' ', price);
    checkItemExistence(name, price);
  };

  const sendPostRequest = (name,price) => {
    const apiUrl = Constants.expoConfig.extra.apiUrl;

    const newItem = {
      name: name,
      price: parseInt(price),
    };
    console.log(newItem);
    fetch(`${apiUrl}/items/`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(newItem),
    })
      .then((response) => {
        if (!response.ok) {
          throw new Error('response not valid');
        }
        return response.json();
      })
      .then((data) => {
        console.log('Item added:', data);
        alert('The item has been added to the basket');
        const ItemId = data.id;
        addLocalItem(ItemId, name, price);
        setItemName('');
        setItemPrice('');
      })
      .catch((error) => {
        console.error('Error requesting the API:', error);
      });
  };

  const addLocalItem = (apiId, name, price) => {
      checkIfLocalItemExists(apiId)
        .then((existingItem) => {
          if (existingItem) {
          console.log('existing')
            // L'article existe déjà, on met à jour la quantité
            const newQuantity = existingItem.quantity + 1;
            db.transaction((tx) => {
              tx.executeSql(
                'UPDATE items SET quantity = ? WHERE id = ?',
                [newQuantity, apiId],
                (_, results) => {
                  console.log(`Item ${name} updated in local db (quantity: ${newQuantity})`);
                },
                (_, error) => {
                  console.log('Error updating item in local database:', error);
                }
              );
            });
          } else {
          console.log('on insere')
            // L'article n'existe pas encore, on insère un nouvel enregistrement avec une quantité initiale de 1
            db.transaction((tx) => {
              tx.executeSql(
                'INSERT INTO items (id, name, price, quantity) VALUES (?, ?, ?, ?)',
                [apiId, name, price, 1], // La quantité initiale est définie à 1
                (_, results) => {
                  console.log(`Item ${name} added in local db (quantity: 1)`);
                },
                (_, error) => {
                  console.log('Error adding item to local database:', error);
                }
              );
            });
          }
        })
        .catch((error) => {
          console.log('Error checking item existence in local database:', error);
        });
    };

  const checkIfLocalItemExists = (apiId) => {
    return new Promise((resolve, reject) => {
      db.transaction((tx) => {
        tx.executeSql(
          'SELECT * FROM items WHERE id = ?',
          [apiId],
          (_, { rows: { _array } }) => {
            if (_array.length > 0) {
              resolve(_array[0]);
            } else {
              resolve(null);
            }
          },
          (_, error) => {
            reject(error);
          }
        );
      });
    });
  };


  const checkItemExistence = async (name, price) => {
    const apiUrl = Constants.expoConfig.extra.apiUrl;
    console.log('item name :', name);

    try {
      const response = await fetch(`${apiUrl}/items/check-existence?item_name=${name}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        }
      });

      if (!response.ok) {
        throw new Error('Response not valid');
      }

      const data = await response.json();

      if (data.exists) {
        alert('Item already exists in the database');

        if (data.item_id) {
          const existingItem = await checkIfLocalItemExists(data.item_id);

          if (existingItem) {
            const newQuantity = existingItem.quantity + 1;

            db.transaction((tx) => {
              tx.executeSql(
                'UPDATE items SET quantity = ? WHERE id = ?',
                [newQuantity, data.item_id],
                (_, results) => {
                  console.log(`Item ${name} updated in local db (quantity: ${newQuantity})`);
                },
                (_, error) => {
                  console.log('Error updating item in local database:', error);
                }
              );
            });
          } else {
            addLocalItem(data.item_id, name, price);
          }
        } else {
          console.error('Item ID is missing in the response');
        }
      } else {
        sendPostRequest(name, price);
      }
    } catch (error) {
      console.error('Error checking item existence:', error);
    }
  };


  const handleBarCodeScanned = ({ type, data }) => {
    setScanned(true);
    alert(`Bar code with type ${type} and data ${data} has been scanned!`);
    setBarcodeData(data);
    extractDataFromBarcode(data);
  };

  const handleAddToBasket = async () => {
    setScanned(true);
    await extractDataFromBarcode(barcodeData);
    sendPostRequest();
  };

  return (
    <View style={Styles.container}>
      {manualEntry === false ? (
        <>
          <Button
            title="Manual Entry"
            onPress={() => setManualEntry(true)}
            style={Styles.topButton}
          />
          <Button
            title="Go to Cart"
            onPress={() => navigation.navigate('CartScreen')}
            style={Styles.topButton}
          />
        </>
      ) : hasPermission === null ? (
        <Text>Requesting camera permission</Text>
      ) : hasPermission === false ? (
        <Text>No access to camera</Text>
      ) : null}
      { manualEntry === true ? (
        <ImageBackground
          source={require('../assets/backCard.jpg')}
          style={{ ...Styles.cardBackground, width: '100%' }}
        >
          <View style={Styles.cardContainer}>
            <View style={Styles.card}>
              <View style={Styles.fieldContainer}>
                <Text>Barcode Data:</Text>
                <TextInput
                  style={Styles.textInput}
                  placeholder="Enter barcode data"
                  value={barcodeData}
                  onChangeText={(text) => setBarcodeData(text)}
                />
              </View>
              <Button title="Add to Basket" onPress={handleAddToBasket} />
              <Button title="Cancel" onPress={() => setManualEntry(false)} />
            </View>
          </View>
        </ImageBackground>
      ) : (
        <View style={Styles.barcodeScannerContainer}>
          <BarCodeScanner
            onBarCodeScanned={scanned ? undefined : handleBarCodeScanned}
            style={StyleSheet.absoluteFillObject}
          />
          {scanned && <Button title={'Tap to Scan Again'} onPress={() => setScanned(false)} />}
        </View>
      )
      }
    </View>
  );

}
