import React, { useEffect, useState } from 'react';
import { View, Text, FlatList, Button } from 'react-native';
import { useNavigation } from '@react-navigation/native';
import Styles from '../styles/Style';
import Icon from 'react-native-vector-icons/MaterialIcons';
import * as SQLite from 'expo-sqlite';
import db from '../database/db';
import Constants from 'expo-constants';

function CartScreen() {
  const navigation = useNavigation();
  const [items, setItems] = useState([]);
  const [totalPrice, setTotalPrice] = useState(0);

  const apiUrl = Constants.expoConfig.extra.apiUrl;

  const getItemsWithQuantities = () => {
    return new Promise((resolve, reject) => {
      db.transaction((tx) => {
        tx.executeSql('SELECT * FROM items', [], (_, { rows }) => {
          const items = rows._array;
          resolve(items);
        });
      });
    });
  };

  useEffect(() => {
    getItemsWithQuantities().then((items) => {
      setItems(items);
      calculateTotalPrice(items);
    });
  }, []);

  const calculateTotalPrice = (items) => {
      const cartItems = items.map((item) => ({
        item_id: item.id,
        quantity: item.quantity,
      }));
      const requestOptions = {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(cartItems),
      };
      fetch(`${apiUrl}/items/calculate-total`, requestOptions)
        .then((response) => {
          if (!response.ok) {
            throw new Error('Response not valid');
          }
          return response.json();
        })
        .then((total) => {
          setTotalPrice((total / 100).toFixed(2));
        })
        .catch((error) => {
          console.error('Error calculating total:', error);
        });
    };


  const renderItem = ({ item }) => (
    <View style={Styles.itemRow}>
      <Text style={Styles.itemName}>{item.name} ({item.quantity})</Text>
      <Text style={Styles.itemPrice}>{(item.price / 100).toFixed(2)} €</Text>
    </View>
  );

  return (
    <View style={Styles.container}>
      <Text style={Styles.title}>
        <Icon name="shopping-basket" size={19} color="black" style={Styles.basketIcon} /> Basket
      </Text>
      <FlatList
        data={items}
        renderItem={renderItem}
        keyExtractor={(item) => item.id.toString()}
      />
      <View style={Styles.totalPriceContainer}>
        <Text style={Styles.totalText}>Total:</Text>
        <Text style={Styles.totalText}>{totalPrice} €</Text>
      </View>
      <Button
        title="Pay"
        onPress={() => navigation.navigate('Checkout', { items: items, totalPrice: totalPrice })}
      />
    </View>
  );
}

export default CartScreen;
