import { useStripe } from "@stripe/stripe-react-native";
import Constants from "expo-constants";
import React, { useEffect, useState } from "react";
import { Alert, Text, Button, SafeAreaView, View } from "react-native";
import BasketScreen from './screens/BasketScreen';
import * as SQLite from 'expo-sqlite';

export default function CheckoutScreen({ route }) {
    const { initPaymentSheet, presentPaymentSheet } = useStripe();
    const [loading, setLoading] = useState(false);
    const [paymentIntentId, setPaymentIntentId] = useState<string>("");

    const apiUrl = Constants.expoConfig.extra.apiUrl;
    const items = route.params.items;

    const userId = "cus_OsU6FAKAAXp1YI";

    const reformulatedList = items.map((item) => ({
        id: item.id,
        amount: item.quantity
    }));

    const db = SQLite.openDatabase('cart.db');

    const fetchPaymentSheetParams = async () => {
        console.log(items);
        const response = await fetch(`${apiUrl}/payments/`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                "pending_items": reformulatedList,
                "customer_id": userId
            })
        });

        const { paymentIntent, ephemeralKey, customer } = await response.json();

        return {
            paymentIntent,
            ephemeralKey,
            customer,
        };
    };

    const initializePaymentSheet = async () => {
        const {
            paymentIntent,
            ephemeralKey,
            customer,
        } = await fetchPaymentSheetParams();

        const { error } = await initPaymentSheet({
            merchantDisplayName: "Example, Inc.",
            customerId: customer,
            customerEphemeralKeySecret: ephemeralKey,
            paymentIntentClientSecret: paymentIntent,
            allowsDelayedPaymentMethods: false,
        });

        if (!error) {
            setPaymentIntentId(paymentIntent);
            setLoading(true);
        }
    };

    const moveItemsToHistory = (itemIds) => {
        db.transaction((tx) => {
            const currentDate = new Date().toISOString();

            itemIds.forEach((itemId) => {
                // we fetch the item details from the "items" table
                tx.executeSql(
                    'SELECT * FROM items WHERE id = ?',
                    [itemId],
                    (_, { rows }) => {
                        const item = rows._array[0];
                        if (item) {
                            // we insert the item into the "history" table with the current date
                            tx.executeSql(
                                'INSERT INTO history (name, price, date) VALUES (?, ?, ?)',
                                [item.name, item.price, currentDate],
                                (_, results) => {
                                    console.log(`Item '${item.name}' moved to history`);
                                },
                                (_, error) => {
                                    console.error('Error inserting item into history:', error);
                                }
                            );
                            // we delete the item from the "items" table
                            tx.executeSql(
                                'DELETE FROM items WHERE id = ?',
                                [itemId],
                                (_, results) => {
                                    console.log(`Item '${item.name}' deleted from items`);
                                },
                                (_, error) => {
                                    console.error('Error deleting item from items:', error);
                                }
                            );
                        }
                    },
                    (error) => {
                        console.error('Error fetching item from items:', error);
                    }
                );
            });
        });
    };

    const openPaymentSheet = async () => {
        const { error } = await presentPaymentSheet();

        if (error) {
            Alert.alert(`Error code: ${error.code}`, error.message);
        } else {
            const paymentIntent = `pi_${paymentIntentId.split("_")[1]}`;
            const response = await fetch(`${apiUrl}/payments/check/${paymentIntent}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    "customer_id": userId
                })
            });

            if (response.status === 200) {
                // we move items to history on successful payment
                moveItemsToHistory(items.map((item) => item.id));
                Alert.alert('Success', 'Your order is confirmed!');
            }
        }
    };

    useEffect(() => {
        initializePaymentSheet();
    }, []);

    return (
        <SafeAreaView>
            <Text>Payment</Text>
            <Button
                disabled={!loading}
                title="Checkout"
                onPress={openPaymentSheet}
            />
        </SafeAreaView>
    );
}
