const { expect } = require("chai");
const { ethers } = require("hardhat");

describe("DufyNFT", function () {
  let contractOwner, user1, user2;
  let dufyNFT;
  const tokenURI = "ipfs://example-uri";
  const fingerprint = "unique-fingerprint";
  const initialPrice = ethers.parseEther("1");
  const royaltyFee = 500; // 5%

  beforeEach(async function () {
    [contractOwner, user1, user2] = await ethers.getSigners();

    const DufyNFTFactory = await ethers.getContractFactory("DufyNFT");
    dufyNFT = await DufyNFTFactory.deploy();
    await dufyNFT.waitForDeployment();
  });

  it("Doit minter un NFT correctement", async function () {
    const tx = await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);
    await expect(tx).to.emit(dufyNFT, "NFTMinted");

    expect(await dufyNFT.totalSupply()).to.equal(1);
    expect(await dufyNFT.ownerOf(1)).to.equal(user1.address);
  });

  it("Doit interdire le mintage par un utilisateur non propriétaire", async function () {
    await expect(
      dufyNFT.connect(user1).mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee)
    ).to.be.revertedWithCustomError(dufyNFT, "OwnableUnauthorizedAccount");
  });

  it("Doit permettre au propriétaire de modifier le prix", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);
    
    const newPrice = ethers.parseEther("2");
    await expect(dufyNFT.connect(user1).setNFTPrice(1, newPrice))
      .to.emit(dufyNFT, "NFTPriceUpdated")
      .withArgs(1, newPrice);

    expect(await dufyNFT.getPrice(1)).to.equal(newPrice);
  });

  it("Doit interdire la modification du prix par un autre utilisateur", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);

    const newPrice = ethers.parseEther("2");
    await expect(dufyNFT.connect(user2).setNFTPrice(1, newPrice)).to.be.revertedWith("Vous n'etes pas le proprietaire");
  });

  it("Doit permettre l'achat d'un NFT", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);

    await expect(
      dufyNFT.connect(user2).buyNFT(1, { value: initialPrice })
    ).to.emit(dufyNFT, "NFTSold")
      .withArgs(1, user2.address, initialPrice);

    expect(await dufyNFT.ownerOf(1)).to.equal(user2.address);
  });

  it("Doit interdire l'achat si le paiement est insuffisant", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);

    await expect(
      dufyNFT.connect(user2).buyNFT(1, { value: ethers.parseEther("0.5") })
    ).to.be.revertedWith("Montant insuffisant");
  });
  
  it("Doit transférer les royalties au créateur lors de la vente", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);

    const balanceBefore = await ethers.provider.getBalance(contractOwner.address);
    await dufyNFT.connect(user2).buyNFT(1, { value: initialPrice });

    const balanceAfter = await ethers.provider.getBalance(contractOwner.address);
    const expectedRoyalty = (initialPrice * BigInt(royaltyFee)) / BigInt(10000);

    expect(balanceAfter - balanceBefore).to.equal(expectedRoyalty);
  });

  it("Doit interdire l'achat d'un NFT inexistant", async function () {
    await expect(
      dufyNFT.connect(user2).buyNFT(99, { value: initialPrice })
    ).to.be.revertedWithCustomError(dufyNFT, "ERC721NonexistentToken");
  });

  it("Doit retourner les bonnes informations avec getNFTData", async function () {
    await dufyNFT.mintNFT(user1.address, tokenURI, fingerprint, initialPrice, royaltyFee);

    const price = await dufyNFT.getPrice(1);
    const retrievedFingerprint = await dufyNFT.tokenFingerprint(1);

    expect(price).to.equal(initialPrice);
    expect(retrievedFingerprint).to.equal(fingerprint);
  });

  it("Doit interdire l'accès aux données d'un NFT inexistant", async function () {
    await expect(dufyNFT.getPrice(99)).to.be.revertedWithCustomError(dufyNFT, "ERC721NonexistentToken");
    await expect(dufyNFT.tokenFingerprint(99)).to.be.revertedWithCustomError(dufyNFT, "ERC721NonexistentToken"); 
  });
});
