import { useAccount, useReadContract, useWriteContract, useWaitForTransactionReceipt } from "wagmi";
import { abi } from "../artifacts/contracts/DufyNFT.sol/DufyNFT.json";
import { useState, useEffect } from "react";
import { ethers } from "ethers";

const contractAddress = "0x5FbDB2315678afecb367f032d93F642f64180aa3"; // Remplacez par l'adresse correcte
const ipfsPrefix = "https://ipfs.io/ipfs"; // Remplacez par votre passerelle IPFS si nécessaire


export default function NFTGallery() {
    const { data: totalNFTs, isLoading, isError, error } = useReadContract({
      address: contractAddress,
      abi,
      functionName: "totalSupply",
    });
  
    const totalNFTsNumber = totalNFTs ? Number(totalNFTs) : 0;
  
    if (isLoading) return (<div className="loader border-t-2 rounded-full border-gray-500 bg-gray-300 animate-spin
                aspect-square w-8 flex justify-center items-center text-yellow-700"></div>);

    if (isError) {
        console.log(error);
        return <p>Erreur lors de la récupération du total des NFTs.</p>;
    }
  
    return (
      <div>
        <h1 className="text-2xl font-bold mb-4 text-center my-2">Total des NFTs disponibles: {totalNFTsNumber}</h1>
        <div className="grid grid-cols-3 gap-4">
          {Array.from({ length: totalNFTsNumber }, (_, i) => (
            <NFTCard key={i} tokenId={i + 1} />
          ))}
        </div>
      </div>
    );
  }
  
export function NFTCard({ tokenId }: { tokenId: number }) {
    const { address, isConnected } = useAccount();
  
    const { data: tokenURI} = useReadContract({
      address: contractAddress,
      abi,
      functionName: "tokenURI",
      args: [tokenId],
    });
  
    const { data: price } = useReadContract({
      address: contractAddress,
      abi,
      functionName: "getPrice",
      args: [tokenId],
    });
  
    const { data: owner } = useReadContract({
      address: contractAddress,
      abi,
      functionName: "ownerOf",
      args: [tokenId],
    });
  
    const { writeContract: buyNFT, data: buyHash, error: buyError, isPending: isBuying } = useWriteContract();
    const { writeContract: setNFTPrice, data: setPriceHash, error: setPriceError, isPending: isSettingPrice } = useWriteContract();
  
    const { isLoading: isBuyConfirming, isSuccess: isBuyConfirmed } = useWaitForTransactionReceipt({ hash: buyHash });
    const { isLoading: isSetPriceConfirming, isSuccess: isSetPriceConfirmed } = useWaitForTransactionReceipt({ hash: setPriceHash });
  
    const [tokenData, setTokenData] = useState<any>(null);
    const [newPrice, setNewPrice] = useState("");
  
    const formattedPrice = price ? ethers.formatEther(price) : "N/A";
    const isOwner = owner?.toLowerCase() === address?.toLowerCase();
  
    useEffect(() => {
      if (!tokenURI) return;
  
      fetch(`${ipfsPrefix}/${tokenURI}`)
        .then((res) => res.json())
        .then((data) => setTokenData(data))
        .catch((err) => console.error("Erreur chargement metadata:", err));
    }, [tokenURI]);
  
    const handleBuy = () => {
      if (!isConnected) {
        alert("Connectez votre wallet pour acheter un NFT.");
        return;
      }
  
      buyNFT({
        address: contractAddress,
        abi,
        functionName: "buyNFT",
        args: [tokenId],
        value: price, // Montant de l'achat
      });
    };
  
    const handleSetPrice = () => {
      if (!isOwner || !newPrice) {
        alert("Vous devez être le propriétaire et entrer un prix valide.");
        return;
      }
  
      setNFTPrice({
        address: contractAddress,
        abi,
        functionName: "setNFTPrice",
        args: [tokenId, ethers.parseEther(newPrice)],
      });
    };
  
    return (
      <div className="border rounded-md p-2 shadow-md">
        {tokenData ? (
          <>
            <img
              src={`${ipfsPrefix}/${tokenData.image}`}
              alt={`NFT ${tokenId}`}
              className="w-full object-cover object-top"
            />
            <p className="text-center mt-2 font-semibold">{tokenData.name}</p>
            <p className="text-center">{tokenData.description}</p>
            <p className="text-center"><span className="font-extrabold">Année :</span> {tokenData.year}</p>
            <p className="text-center font-bold mt-2"><span className="font-extrabold">Prix :</span> {formattedPrice} ETH</p>
  
            {isOwner ? (
              <div className="text-center mt-4">
                <input
                  type="number"
                  className="border p-1 mr-2"
                  placeholder="Nouveau prix"
                  value={newPrice}
                  onChange={(e) => setNewPrice(e.target.value)}
                />
                <button
                  onClick={handleSetPrice}
                  disabled={isSettingPrice || isSetPriceConfirming}
                  className="border px-4 py-2 rounded bg-green-500 text-white hover:bg-green-700"
                >
                  {isSettingPrice || isSetPriceConfirming ? "Mise à jour..." : "Modifier Prix"}
                </button>
                {isSetPriceConfirmed && <p className="text-green-500 mt-2">Prix mis à jour ✅</p>}
                {setPriceError && <p className="text-red-500 mt-2">Erreur: {setPriceError.message}</p>}
              </div>
            ) : (
              <button
                onClick={handleBuy}
                disabled={isBuying || isBuyConfirming}
                className="border px-5 py-2 rounded mt-4 bg-blue-500 text-white hover:bg-blue-700"
              >
                {isBuying || isBuyConfirming ? "Achat en cours..." : "Acheter"}
              </button>
            )}
            {isBuyConfirmed && <p className="text-green-500 mt-2">NFT acheté ✅</p>}
            {buyError && <p className="text-red-500 mt-2">Erreur: {buyError.message}</p>}
          </>
        ) : (
          <div className="flex flex-col bg-neutral-300 w-56 h-64 animate-pulse rounded-xl p-4 gap-4">
            <div className="bg-neutral-400/50 w-full h-32 animate-pulse rounded-md"></div>
            <div className="flex flex-col gap-2">
                <div className="bg-neutral-400/50 w-full h-4 animate-pulse rounded-md"></div>
                <div className="bg-neutral-400/50 w-4/5 h-4 animate-pulse rounded-md"></div>
                <div className="bg-neutral-400/50 w-full h-4 animate-pulse rounded-md"></div>
                <div className="bg-neutral-400/50 w-2/4 h-4 animate-pulse rounded-md"></div>
            </div>
          </div>
        )}
      </div>
    );
  }