// SPDX-License-Identifier: MIT
pragma solidity ^0.8.28;

import "@openzeppelin/contracts/token/ERC721/extensions/ERC721URIStorage.sol";
import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/token/common/ERC2981.sol";
import "@openzeppelin/contracts/utils/ReentrancyGuard.sol";

contract DufyNFT is ERC721URIStorage, ERC2981, Ownable, ReentrancyGuard {
    uint256 private _tokenIds;
    address private immutable _royaltyReceiver;

    mapping(uint256 => uint96) private _tokenRoyalties;
    mapping(uint256 => string) private _fingerprints;
    mapping(uint256 => uint256) public nftPrices;

    event NFTMinted(uint256 indexed tokenId, address indexed recipient, uint256 price, uint96 royalty);
    event NFTPriceUpdated(uint256 indexed tokenId, uint256 newPrice);
    event NFTSold(uint256 indexed tokenId, address indexed buyer, uint256 price);

    constructor() ERC721("DufyNFT", "DUFY") Ownable(msg.sender) {
        _royaltyReceiver = msg.sender;
    }

    /**
     * @dev Mint un NFT avec un taux de royalties dynamique (max 10%).
     */
    function mintNFT(
        address recipient,
        string memory tokenURI,
        string memory fingerprint,
        uint256 price,
        uint96 royaltyFee
    ) public onlyOwner returns (uint256) {
        require(royaltyFee <= 1000, "Royalties trop elevees (max 10%)");
        require(recipient != address(0), "Recipient invalide");

        _tokenIds++;
        uint256 newItemId = _tokenIds;

        _mint(recipient, newItemId);
        _setTokenURI(newItemId, tokenURI);
        _fingerprints[newItemId] = fingerprint;
        nftPrices[newItemId] = price;
        _tokenRoyalties[newItemId] = royaltyFee;

        _setTokenRoyalty(newItemId, _royaltyReceiver, royaltyFee);

        emit NFTMinted(newItemId, recipient, price, royaltyFee);
        return newItemId;
    }

    /**
     * @dev Retourne les informations sur les royalties pour chaque NFT.
     */
    function royaltyInfo(uint256 tokenId, uint256 salePrice)
        public
        view
        override
        returns (address receiver, uint256 royaltyAmount)
    {
        require(ownerOf(tokenId) != address(0), "NFT inexistant");
        uint96 royaltyFee = _tokenRoyalties[tokenId];
        return (_royaltyReceiver, (salePrice * royaltyFee) / 10000);
    }

    /**
     * @dev Permet au propriétaire d'un NFT de fixer un prix pour la vente.
     */
    function setNFTPrice(uint256 tokenId, uint256 price) public {
        require(ownerOf(tokenId) == msg.sender, "Vous n'etes pas le proprietaire");
        nftPrices[tokenId] = price;

        emit NFTPriceUpdated(tokenId, price);
    }

    /**
     * @dev Permet d'acheter un NFT.
     */
    function buyNFT(uint256 tokenId) public payable nonReentrant {
        address seller = ownerOf(tokenId);
        uint256 price = nftPrices[tokenId];
        require(ownerOf(tokenId) != address(0), "NFT inexistant");
        require(price > 0, "Ce NFT n'est pas a vendre");
        require(msg.value >= price, "Montant insuffisant");

        uint96 royaltyFee = _tokenRoyalties[tokenId];
        uint256 royalties = (price * royaltyFee) / 10000;
        uint256 sellerProceeds = price - royalties;

        if (royalties > 0) {
            (bool royaltyPaid, ) = payable(_royaltyReceiver).call{value: royalties}("");
            require(royaltyPaid, "Echec du transfert des royalties");
        }

        (bool sellerPaid, ) = payable(seller).call{value: sellerProceeds}("");
        require(sellerPaid, "Echec du transfert au vendeur");

        _transfer(seller, msg.sender, tokenId);
        nftPrices[tokenId] = 0;

        emit NFTSold(tokenId, msg.sender, price);
    }

    /**
     * @dev Retourne le prix d'un NFT spécifique.
     */
    function getPrice(uint256 tokenId) public view returns (uint256) {
        require(ownerOf(tokenId) != address(0), "Token inexistant");
        return nftPrices[tokenId];
    }

    /**
     * @dev Retourne le fingerprint du NFT.
     */
    function tokenFingerprint(uint256 tokenId) public view returns (string memory) {
        require(ownerOf(tokenId) != address(0), "Token inexistant");
        return _fingerprints[tokenId];
    }

    /**
     * @dev Retourne le nombre total de NFTs mintés.
     */
    function totalSupply() external view returns (uint256) {
        return _tokenIds;
    }

    /**
     * @dev Résolution des conflits de `supportsInterface` avec `ERC721URIStorage` et `ERC2981`.
     */
    function supportsInterface(bytes4 interfaceId)
        public
        view
        override(ERC721URIStorage, ERC2981)
        returns (bool)
    {
        return super.supportsInterface(interfaceId);
    }
}
