
import React, { useEffect, useState, useRef } from "react";
import * as d3 from "d3";
import { nice, ticks } from "d3";
import { AxisX, AxisY } from './Axis';

const dateFormatOptions:Intl.DateTimeFormatOptions = {month: "long", year: "numeric"}

const textLabel = <>
    <tspan x="0" dy="1.2em">Nombre</tspan>
    <tspan x="0" dy="1.2em">de</tspan>
    <tspan x="0" dy="1.2em">ventes</tspan>
</>

export default function Graph2() {
    const Y_MAX = 500;
    const X_MAX = 1200;
    const MARGIN_LEFT = 150;
    const MARGIN_TOP = 50;

    const [xTicks, setXTicks] = useState<{ value: Date; xOffset: number; }[]>([])
    const [yTicks, setYTicks] = useState<{value: number, yOffset: number}[]>([])
    const [data,setData] = useState<{ date: Date; xOffset: number; amount: number, width: number, height: number}[]>([])

    useEffect(() => {
        const items = [
            { dateBegin: "2017-01-01", dateEnd: "2017-02-01", amount: 50 }, 
            { dateBegin: "2017-06-01", dateEnd: "2017-07-01", amount: 30 }, 
            { dateBegin: "2018-01-01", dateEnd: "2018-02-01", amount: 70 }]

        const xScale = d3.scaleTime()
            .domain([new Date(items[0].dateBegin), new Date(items[2].dateEnd)])
            .nice()
            .range([50, X_MAX])

        //On récupère les ticks de xScale, en enlevant le dernier élément en trop
        let tmpTicks = xScale.ticks()
            .map(value => ({
                value,
                xOffset: xScale(value)
            }))
        tmpTicks = tmpTicks.slice(0, -1)

        setXTicks(tmpTicks)
        
        let sortedItems = items.sort((a,b) => {return a.amount - b.amount})
        
        const yScale = d3.scaleLinear()
            .domain([sortedItems[sortedItems.length - 1].amount, 0])
            .nice()
            .range([0, Y_MAX])
        
        setYTicks(yScale.ticks()
            .map(value => ({
                value,
                yOffset: yScale(value)
            })))

        setData(items.map(item => {
            let startDate = new Date(item.dateBegin)
            let endDate = new Date(item.dateEnd)
            return {
                date: startDate, 
                xOffset: xScale(startDate), 
                amount: item.amount, 
                height: Y_MAX - yScale(item.amount),
                width: xScale(endDate) - xScale(startDate)
            }
        }));
    },[])

    return (
        <div>
            <svg id="graph"
                width="100%"
                height={Y_MAX + MARGIN_TOP + 30}
            >
                {/* Définition du motif d'hachure pour les rectangles */}
                <defs>
                    <pattern
                            id="diagonalHatch"
                            patternUnits="userSpaceOnUse" 
                            width="4" 
                            height="8" 
                            patternTransform="rotate(-45 2 2)"
                        >
                            <path d="M -1,2 l 6,0" stroke="red" strokeWidth=".5"/>
                    </pattern>
                </defs>
                <AxisX ticks={xTicks} transform={`translate(${MARGIN_LEFT}, ${Y_MAX + MARGIN_TOP})`} dateFormat={dateFormatOptions}/>
                <AxisY ticks={yTicks} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} textLabel={textLabel}/>
                <BarChart data={data} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} yMax={Y_MAX}/>
            </svg>
        </div>
    )
}

/**
 * Fonction dessinant l'histogramme
 * @param param0 
 * @returns 
 */
function BarChart({data, transform, yMax}: {data:{ date: Date; xOffset: number; amount: number, width: number, height: number}[], transform:string, yMax:number}){
    return (
        <g
            key="barChart"
            transform={transform}
        >
            {data.map(({ date, xOffset, width, height }) => (
                <g
                    key={"data:"+date.toISOString()}
                    transform={`translate(${xOffset}, ${yMax - height})`}
                >
                    <rect 
                        width={width}
                        height={height}
                        stroke="red"
                        fill="url(#diagonalHatch)"
                    />
                </g>
            ))}
        </g>
    )
}