
import React, { useEffect, useState, useRef } from "react";
import * as d3 from "d3";
import { nice, text, ticks } from "d3";
import { AxisX, AxisY } from './Axis';

const dateFormatOptions:Intl.DateTimeFormatOptions = {year: "numeric"}

const textLabel = <>
    <tspan x="0" dy="1.2em">Prix</tspan>
    <tspan x="0" dy="1.2em">du m²</tspan>
</>

export default function Graph1() {
    const Y_MAX = 500;
    const X_MAX = 1200;
    const MARGIN_LEFT = 150;
    const MARGIN_TOP = 50;

    const [xTicks, setXTicks] = useState<{ value: Date; xOffset: number; }[]>([])
    const [yTicks, setYTicks] = useState<{value: number, yOffset: number}[]>([])
    const [data,setData] = useState<{ date: Date; yOffset: number; xOffset: number; amount: number, width: number, height: number}[]>([])
    const [line, setLinePath] = useState<string>("");

    useEffect(() => {
        const items = [
            { dateBegin: "2016-01-01", dateEnd: "2016-02-01", amount: 50 }, 
            { dateBegin: "2017-01-01", dateEnd: "2017-02-01", amount: 30 }, 
            { dateBegin: "2018-01-01", dateEnd: "2018-02-01", amount: 70 },
            { dateBegin: "2019-01-01", dateEnd: "2019-02-01", amount: 50 }]

        const xScale = d3.scaleTime()
            .domain([new Date(items[0].dateBegin), new Date(items[items.length-1].dateEnd)])
            .nice()
            .range([50, X_MAX])

        //On récupère les ticks de xScale, en enlevant le dernier élément en trop
        let tmpTicks = xScale.ticks(items.length)
            .map(value => ({
                value,
                xOffset: xScale(value)
            }))

        setXTicks(tmpTicks)
        
        let sortedItems = items.sort((a,b) => {return a.amount - b.amount})
        
        const yScale = d3.scaleLinear()
            .domain([sortedItems[sortedItems.length - 1].amount, 0])
            .nice()
            .range([0, Y_MAX])
        
        setYTicks(yScale.ticks()
            .map(value => ({
                value,
                yOffset: yScale(value)
            })))

        setData(items.map(item => {
            let startDate = new Date(item.dateBegin)
            let endDate = new Date(item.dateEnd)
            return {
                date: startDate, 
                xOffset: xScale(startDate),
                yOffset: yScale(item.amount), 
                amount: item.amount, 
                height: Y_MAX - yScale(item.amount),
                width: xScale(endDate) - xScale(startDate)
            }
        }));

        const dataset:[number,number][] = items.map((d) => ([
            Math.round(xScale(new Date(d.dateBegin))),
            Math.round(yScale(d.amount))
        ]));

        dataset.sort((a,b) => a[0] - b[0])

        const line = d3
            .line()
            .curve(d3.curveMonotoneX)(dataset) || "";

        setLinePath(line);
        console.log(line)
    },[])

    return (
        <div>
            <svg id="graph"
                width="100%"
                height={Y_MAX + MARGIN_TOP + 30}
            >
                <AxisX ticks={xTicks} transform={`translate(${MARGIN_LEFT}, ${Y_MAX + MARGIN_TOP})`} dateFormat={dateFormatOptions}/>
                <AxisY ticks={yTicks} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} textLabel={textLabel}/>
                <LineChart data={data} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} linePath={line}/>
            </svg>
        </div>
    )
}

/**
 * Fonction dessinant l'histogramme
 * @param param0 
 * @returns 
 */
function LineChart({data, transform, linePath}: {data:{ date: Date; yOffset: number; xOffset: number; amount: number, width: number, height: number}[], transform:string, linePath:string}){
    return (
        <g
            key="lineChart"
            transform={transform}
        >
            {data.map(({ date, yOffset, xOffset, width, height }) => (
                <circle cx={xOffset} cy={yOffset} key={"data:"+date.toISOString()} r={5} color={"black"} />
            ))}
            <path strokeWidth={3} fill="none" stroke={"black"} d={linePath} />
        </g>
    )
}