
import React, { useEffect, useState } from "react";
import * as d3 from "d3";
import {AxisX, AxisY, getScales} from '../../components/d3js/Axis';
import {ApiService} from "../../components/d3js/ApiService";
import {TooltipDivs} from "../../components/d3js/Legends"
import 'react-tooltip/dist/react-tooltip.css';
import Spinner from "../../components/d3js/Spinner";
import { primaryColor, secondaryColor } from "../../components/common/colors";
import {randomKey} from "../../components/common/randomKey";
import {resizeGraph} from "../../components/d3js/resizeGraph";

const textLabel = <>
    <tspan x="0" dy="1.2em">Prix</tspan>
    <tspan x="0" dy="1.2em">du m²</tspan>
</>

export default function Series() {
    const Y_MAX = 500;
    const X_MAX = 1200;
    const MARGIN_LEFT = 150;
    const MARGIN_TOP = 50;

    const [items, setItems] = useState<Series[]>([]);
    const [xTicks, setXTicks] = useState<XTicks[]>([]);
    const [yTicks, setYTicks] = useState<YTicks[]>([]);
    const [divWidth, setDivWidth] = useState(-1);
    const [data, setData] = useState<SeriesData[]>([])
    const [line, setLinePath] = useState<string>("");

    useEffect(() => {
        window.addEventListener("resize", () => {
            resizeGraph("seriesGraph", setDivWidth);
        })
        resizeGraph("seriesGraph", setDivWidth);

        ApiService.getSeries()
            .then(series => setItems(series));
    }, [])

    useEffect(() => {
        if(items.length <= 0)
            return ;

        const {xScale, yScale} = getScales(items, new Date(items[0].date), new Date(items[items.length-1].date), divWidth>0?divWidth-200:X_MAX, Y_MAX)

        setXTicks(xScale.ticks()
            .map(value => ({
                value,
                xOffset: xScale(value)
            })))
        
        setYTicks(yScale.ticks()
            .map(value => ({
                value,
                yOffset: yScale(value)
            })))

        setData(items.map(item => {
            const date = new Date(item.date)
            return {
                date: date,
                xOffset: xScale(date),
                yOffset: yScale(item.value),
                amount: item.value,
                height: Y_MAX - yScale(item.value)
            }
        }));

        const dataset:[number,number][] = items.map((d) => ([
            Math.round(xScale(new Date(d.date))),
            Math.round(yScale(d.value))
        ]));

        dataset.sort((a,b) => a[0] - b[0])

        const line = d3
            .line()
            .curve(d3.curveMonotoneX)(dataset) || "";

        setLinePath(line);
    },[items, divWidth])

    return (
        <>
            <Spinner active={items.length <= 0} />
            <div id="seriesGraph" data-testid={"seriesGraph"}>
                {items.length > 0 &&
                <>
                    <svg id="graph"
                        width="100%"
                        height={Y_MAX + MARGIN_TOP + 30}
                    >
                        <AxisX ticks={xTicks} transform={`translate(${MARGIN_LEFT}, ${Y_MAX + MARGIN_TOP})`} />
                        <AxisY ticks={yTicks} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} textLabel={textLabel}/>
                        <LineChart data={data} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} linePath={line}/>
                    </svg>
                    <TooltipDivs data={data} />
                </>
                }
            </div>
        </>
    )
}

/**
 * Fonction dessinant l'histogramme
 * @param param0 
 * @returns 
 */
function LineChart({data, transform, linePath}: {data: SeriesData[], transform:string, linePath:string}){
    return (
        <g
            key="lineChart"
            transform={transform}
        >
            <path strokeWidth={3} fill="none" stroke={primaryColor} d={linePath} />
            {data.map(({ date, yOffset, xOffset, height, amount }) => (
                <circle
                    cx={xOffset}
                    cy={yOffset}
                    key={randomKey()}
                    id={"data:"+date.toISOString()}
                    data-testid={"data:"+date.toISOString()}
                    r={5}
                    fill={secondaryColor}
                    data-tooltip-content={`Prix médian : ${Intl.NumberFormat('fr-FR').format(amount)}€`}
                    data-tooltip-place="top"
                />
            ))}
        </g>
    )
}