
import React, {ChangeEvent, useEffect, useState} from "react";
import {AxisX, AxisY, getScales} from '../../components/d3js/Axis';
import {ApiService} from "../../components/d3js/ApiService";
import {TooltipDivs} from "../../components/d3js/Legends";
import 'react-tooltip/dist/react-tooltip.css';
import Spinner from "../../components/d3js/Spinner";
import { primaryColor, secondaryColor } from "../../components/common/colors";
import {randomKey} from "../../components/common/randomKey";
import {resizeGraph} from "../../components/d3js/resizeGraph";
import { useDebounce } from "../../components/common/hooks";

const textLabel = <>
    <tspan x="0" dy="1.2em">Nombre</tspan>
    <tspan x="0" dy="1.2em">de</tspan>
    <tspan x="0" dy="1.2em">ventes</tspan>
</>

const groups = [
    { group: "day", label: "jours" },
    { group: "week", label: "semaines" },
    { group: "month", label: "mois" },
    { group: "year", label: "années" },
];
const defaults = {
    groupBy: groups[2],
    dateStart: new Date("2018-01-01"),
    dateEnd: new Date("2019-01-01")
};
const minDateString = "2018-01-01";
const maxDateString = "2022-01-01";
const minDate = new Date(minDateString);
const maxDate = new Date(maxDateString);

export default function Amounts() {
    const Y_MAX = 500;
    const X_MAX = 1250;
    const MARGIN_LEFT = 150;
    const MARGIN_TOP = 50;

    const [items, setItems] = useState<Amount[]>([])
    const [xTicks, setXTicks] = useState<XTicks[]>([])
    const [yTicks, setYTicks] = useState<YTicks[]>([])
    const [data, setData] = useState<AmountData[]>([])
    const [divWidth, setDivWidth] = useState(-1);
    const [groupeBy, setGroupeBy] = useState(defaults.groupBy.group);
    const [dateDebut, setDateDebut] = useState(defaults.dateStart)
    const [dateFin, setDateFin] = useState(defaults.dateEnd);
    const dateDebutDebounce = useDebounce(dateDebut, 250);
    const dateFinDebounce = useDebounce(dateFin, 250);

    /**
     * Fonction permettant de récupérer la date de fin d'un item, par rapport au groupement et aux bornes
     * @param date
     */
    function getEndDate(date: Date){
        const endDate = new Date(date);
        switch(groupeBy){
            // Si le groupement est par jour, la valeur est valable jusqu'au jour suivant
            case groups[0].group:
                endDate.setDate(endDate.getDate()+1);
                break;
            // Si le groupement est par semaine, la valeur est valable jusqu'au lundi suivant
            case groups[1].group:
                endDate.setDate(endDate.getDate() - endDate.getDay() + 7 + 1);
                break;
            // Si le groupement est par mois, la valeur est valable jusqu'au premier jour du mois suivant
            case groups[2].group:
                endDate.setFullYear(endDate.getFullYear(), endDate.getMonth()+1, 1);
                break;
            // Si le groupement est par année, la valeur est valable jusqu'au 01/01 de l'année suivante
            case groups[3].group:
                endDate.setFullYear(endDate.getFullYear()+1, 0, 1);
                break;
        }

        // Renvoie la date la plus proche entre celle calculée et la borne de fin
        // @ts-ignore
        return new Date(Math.min(endDate, new Date(dateFin)));
    }

    const getDateInput = (e: ChangeEvent<HTMLInputElement>) => {
        const { value } = e.target;
        const date = new Date(value);
        if (isNaN(date.getTime())) {
            // @ts-ignore
            const initialDate: Date = defaults[e.target.id];
            return initialDate;
        }

        if (date < minDate)
            return minDate;

        if (date > maxDate)
            return maxDate;

        return date;
    };

    const handleDateDebutChanged = (e: ChangeEvent<HTMLInputElement>) => {
        const date = getDateInput(e);
        if (date > dateFin) {
            setDateFin(date);
        }
        setDateDebut(date);
    };

    const handleDateFinChanged = (e: ChangeEvent<HTMLInputElement>) => {
        const date = getDateInput(e);
        if (date < dateDebut) {
            setDateDebut(date);
        }
        setDateFin(date);
    };

    //Gère la largeur du graphique en fonction de la taille de la fenêtre
    useEffect(() => {
        window.addEventListener("resize", () => {
            resizeGraph("salesGraph", setDivWidth);
        })
        resizeGraph("salesGraph", setDivWidth);
    }, [])

    //Met a jour les données lorsque la taille du groupe change, ou lorsque l'utilisateur change la date de début / de fin
    useEffect(() => {
        ApiService.getAmounts(dateDebut, dateFin, groupeBy)
            .then(amounts => {
                setItems(amounts);
            });
    }, [dateDebutDebounce, dateFinDebounce, groupeBy])

    //Met a jour le graph lorsque les données changent, ou lorsqu'on change la taille de la fenêtre
    useEffect(() => {
        if(!items || items.length === 0) {
            setData([]);
            return;
        }

        const {xScale, yScale} = getScales(items, dateDebut, dateFin, divWidth > 0?divWidth-250:X_MAX, Y_MAX);

        // Abscisse
        setXTicks(xScale.ticks()
            .map(value => ({
                value,
                xOffset: xScale(value)
            })))

        // Ordonée
        setYTicks(yScale.ticks()
            .map(value => ({
                value,
                yOffset: yScale(value)
            })))

        // Données
        setData(items.map((item) => {
            //Si la date de début est inférieur à la borne de début, on prend la borne à la place
            // @ts-ignore
            let startDate = new Date(Math.max(new Date(item.date), new Date(dateDebut)));
            let endDate = getEndDate(startDate);
            return {
                date: startDate,
                xOffset: xScale(startDate),
                amount: item.value,
                height: Y_MAX - yScale(item.value),
                width: xScale(endDate) - xScale(startDate)
            }
        }));
    }, [items, divWidth, dateDebutDebounce, dateFinDebounce, groupeBy])

    return (
        <>
            <div className="container-md">
                <div className="row mb-2 justify-content-center">
                    <label htmlFor="selectGroup" className="col-form-label col-md-2 text-end">ventes par</label>
                    <div className="col-md-4">
                        <select
                            data-testid="selectGroupBy"
                            defaultValue={groupeBy}
                            onChange={(e) => setGroupeBy(e.target.value)}
                            id="selectGroup" className="form-select ms-2"
                        >
                            {groups.map(({group, label}) => (
                                <option
                                    key={group}
                                    value={group}
                                >{label}</option>
                            ))}
                        </select>
                    </div>
                </div>
                <div className="row mb-2 justify-content-center">
                    <label htmlFor="dateStart" className="col-form-label col-md-2 text-end">entre le</label>
                    <div className="col-md-4">
                        <input id="dateStart" className="form-control ms-2" type="date"
                           data-testid="dateDebut"
                            value={dateDebut.toISOString().substring(0, dateDebut.toISOString().indexOf("T"))} onChange={handleDateDebutChanged}
                            min={minDateString} max={maxDateString} />
                    </div>
                </div>
                <div className="row mb-2 justify-content-center">
                    <label htmlFor="dateEnd" className="col-form-label col-md-2 text-end">et le</label>
                    <div className="col-md-4">
                        <input id="dateEnd" className="form-control ms-2" type="date"
                           data-testid="dateFin"
                            value={dateFin.toISOString().substring(0, dateFin.toISOString().indexOf("T"))} onChange={handleDateFinChanged}
                            min={minDateString} max={maxDateString} />
                    </div>
                </div>
            </div>
            <Spinner active={(items.length <= 0) != (dateDebut.getTime() === dateFin.getTime())} />
            <div id="salesGraph">
                {(data.length > 0 || dateDebut.getTime() === dateFin.getTime()) &&
                    <>
                        <svg id="graph"
                             width="100%"
                             height={Y_MAX + MARGIN_TOP + 30}
                             data-testid={"svg"}
                        >
                            {/* Définition du motif d'hachure pour les rectangles */}
                            <defs>
                                <pattern
                                    id="diagonalHatch"
                                    patternUnits="userSpaceOnUse"
                                    width="4"
                                    height="8"
                                    patternTransform="rotate(-45 2 2)"
                                >
                                    <path d="M -1,2 l 6,0" stroke={secondaryColor} strokeWidth=".5"/>
                                </pattern>
                            </defs>
                            <AxisX ticks={xTicks} transform={`translate(${MARGIN_LEFT}, ${Y_MAX + MARGIN_TOP})`}/>
                            <AxisY ticks={yTicks} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} textLabel={textLabel}/>
                            <BarChart data={data} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} yMax={Y_MAX}/>
                        </svg>
                        <TooltipDivs data={data} />
                    </>}
            </div>
        </>
    )
}

/**
 * Fonction dessinant l'histogramme
 * @param param0 
 * @returns 
 */
function BarChart({data, transform, yMax}: {data: AmountData[], transform:string, yMax:number}){
    return (
        <g
            key="barChart"
            transform={transform}
        >
            {data.map(({ date, xOffset, width, height, amount }) => (
                <g
                    key={randomKey()}
                    id={"data:"+date.toISOString()}
                    data-testid={"data:"+date.toISOString()}
                    transform={`translate(${xOffset}, ${yMax - height})`}
                    // Le tooltip affiche la date de début et la valeur
                    data-tooltip-html={`Date : ${Intl.DateTimeFormat('fr-FR').format(date)} \
                                        <br /> Nombre de ventes : ${Intl.NumberFormat('fr-FR').format(amount)}`}
                    data-tooltip-place="top"
                >
                    <rect 
                        width={width}
                        height={height}
                        stroke={primaryColor}
                        fill="url(#diagonalHatch)"
                    />
                </g>
            ))}
        </g>
    )
}