import {
    render,
    screen,
    cleanup,
    waitFor,
    fireEvent
} from '@testing-library/react'
import '@testing-library/jest-dom'
import fetch from 'node-fetch';
import {act} from "react-dom/test-utils";
import Regions from "../pages/regions";
import {ApiService} from "../components/d3js/ApiService";

// Test items
const items = [
    {region: "Bretagne", percentage: 5},
    {region: "Nouvelle Aquitaine", percentage: 22},
    {region: "Occitanie", percentage: 16},
    {region: "Centre Val de Loire", percentage: 8},
    {region: "Normandie", percentage: 11.5},
    {region: "Ile de France", percentage: 13},
    {region: "Grand Est", percentage: 12}
]

jest.mock('node-fetch');

describe('Tests sur le graphe du nombre de ventes', () => {
    afterEach(() => {
        jest.clearAllMocks()
        cleanup();
    })
    describe('Tests sans données', () => {
        beforeEach(() => {
            jest.spyOn(ApiService, "getRegions").mockReturnValue(Promise.resolve([]));
        })
        test('Page de base', async () => {
            act(() => {
                render(<Regions />)
            })
            await waitFor(() => {
                const spinner = screen.getByTestId("spinner");
                expect(spinner).toBeInTheDocument();
            })
        })
    })
    describe('Tests avec des données', () => {
        beforeEach(() => {
            jest.spyOn(ApiService, "getRegions").mockReturnValue(Promise.resolve(items));
        })
        test('Page de base', async () => {
            act(() => {
                render(<Regions />)
            })
            await waitFor(() => {
                for(let item of items){
                    let itemG = screen.getByTestId(`data:${item.region}`);
                    expect(itemG).toBeInTheDocument();
                    const text = itemG.querySelector("text");
                    const percentage = parseFloat(text.textContent.substring(0, text.textContent.indexOf(" ")));
                    expect(percentage).toBe(item.percentage);
                }
                let total = items.map(item => item.percentage).reduce((partialSum, a) => partialSum + a, 0)
                const autre = screen.getByTestId(/data:Autre/i);
                expect(autre).toBeInTheDocument();
                const text = autre.querySelector("text");
                const percentage = parseFloat(text.textContent.substring(0, text.textContent.indexOf(" ")));
                expect(percentage).toBe(100 - total);
            })
        })
        test('Changement d\'année', async () => {
            const newItems = [
                {region: "Bretagne", percentage: 10},
                {region: "Nouvelle Aquitaine", percentage: 15},
                {region: "Occitanie", percentage: 5},
                {region: "Centre Val de Loire", percentage: 3.5},
                {region: "Normandie", percentage: 21},
                {region: "Ile de France", percentage: 13},
                {region: "Grand Est", percentage: 4.5}
            ]
            act(() => {
                render(<Regions />)
            })
            const input = await screen.findByTestId("input:2018");
            act(() => {
                jest.spyOn(ApiService, "getRegions").mockReturnValue(Promise.resolve(newItems));
                fireEvent.click(input);
            })
            await waitFor(() => {
                for(let item of newItems){
                    let itemG = screen.getByTestId(`data:${item.region}`);
                    expect(itemG).toBeInTheDocument();
                    const text = itemG.querySelector("text");
                    const percentage = parseFloat(text.textContent.substring(0, text.textContent.indexOf(" ")));
                    expect(percentage).toBe(item.percentage);
                }
                let total = newItems.map(item => item.percentage).reduce((partialSum, a) => partialSum + a, 0)
                const autre = screen.getByTestId(/data:Autre/i);
                expect(autre).toBeInTheDocument();
                const text = autre.querySelector("text");
                const percentage = parseFloat(text.textContent.substring(0, text.textContent.indexOf(" ")));
                expect(percentage).toBe(100 - total);
            })
        })
    })
})