import {
    render,
    screen,
    cleanup,
    waitFor,
    fireEvent
} from '@testing-library/react'
import '@testing-library/jest-dom'
import Amounts from "../pages/amount";
import {act} from "react-dom/test-utils";
import {ApiService} from "../components/d3js/ApiService";

const items = [
    {date: '2018-01-01', value: 50},
    {date: '2018-02-01', value: 40},
    {date: '2018-03-01', value: 45},
    {date: '2018-04-01', value: 35},
    {date: '2018-05-01', value: 55},
    {date: '2018-06-01', value: 30},
    {date: '2018-07-01', value: 120},
    {date: '2018-08-01', value: 155},
    {date: '2018-09-01', value: 130},
    {date: '2018-10-01', value: 200},
    {date: '2018-11-01', value: 250},
    {date: '2018-12-01', value: 20},
]

describe('Tests sur le graphe des régions', () => {
    afterEach(() => {
        jest.clearAllMocks()
        cleanup();
    })
    describe('Test sans données', () => {
        beforeEach(() => {
            jest.spyOn(ApiService, "getAmounts").mockReturnValue(Promise.resolve([]));
        })
        test('Page de base', async () => {
            act(() => {
                render(<Amounts />)
            })
            await waitFor(() => {
                const spinner = screen.getByTestId("spinner");
                expect(spinner).toBeInTheDocument();
            })
        })
    })
    describe('Test avec des données', () => {
        beforeEach(() => {
            jest.spyOn(ApiService, "getAmounts").mockReturnValue(Promise.resolve(items));
        })
        test('Page de base', async () => {
            act(() => {
                render(<Amounts />)
            })
            await waitFor(() => {
                for(let item of items){
                    let itemG = screen.getByTestId(`data:${item.date}T00:00:00.000Z`);
                    expect(itemG).toBeInTheDocument();
                }
            })
        })
        describe('Test sur le changement des dates', () => {
            test('Changement de la date de début', async () => {
                act(() => {
                    render(<Amounts />)
                })
                await act(async () => {
                    const dateDebut = screen.getByTestId("dateDebut");
                    fireEvent.change(dateDebut, {target: {value: "2018-02-01"}});
                })

                await waitFor(async () => {
                    const firstDateScale = await screen.findByTestId(/xAxis:0/i);
                    expect(firstDateScale).toHaveTextContent("février 2018");
                })
            })
            test('Changement de la date de fin', async () => {
                act(() => {
                    render(<Amounts />)
                })

                let ticks = await screen.findAllByTestId(/xAxis:/i);
                const size = ticks.length;

                await act(async () => {
                    const dateFin = screen.getByTestId("dateFin");
                    fireEvent.change(dateFin, {target: {value: "2018-12-01"}});
                })

                await waitFor(async () => {
                    let ticks = screen.getAllByTestId(/xAxis:/);
                    expect(ticks).toHaveLength(size-1);
                })
            })
            test('Changement de la date invalide', async () => {
                act(() => {
                    render(<Amounts />)
                })

                const dateDebut = screen.getByTestId("dateDebut");
                const dateFin = screen.getByTestId("dateFin");
                await act(async () => {
                    fireEvent.change(dateDebut, {target: {value: "yyyy-mm-dd"}});
                    fireEvent.change(dateDebut, {target: {value: "2000-01-01"}});
                    fireEvent.change(dateDebut, {target: {value: "2020-01-01"}});
                    fireEvent.change(dateFin, {target: {value: "2018-01-01"}});
                    fireEvent.change(dateFin, {target: {value: "2050-01-01"}});
                })

                await waitFor(async () => {
                    expect(dateDebut.value).toEqual('2018-01-01');
                    expect(dateFin.value).toEqual('2022-01-01');
                })
            });
        })
        describe('Test sur les changements de groupes', () => {
            test('GroupBy semaine', async () => {
                act(() => {
                    render(<Amounts />)
                })

                const g = await screen.findAllByTestId(/data:/i);
                const rectsWidth = g.map(item => parseFloat(item.querySelector("rect").getAttribute("width")))

                await act(async () => {
                    const select = screen.getByTestId("selectGroupBy");
                    fireEvent.change(select, {target: {value: "week"}});
                })

                await waitFor(() => {
                    const itemsG = screen.getAllByTestId(/data:/i)
                    for(let index in itemsG){
                        const rectWidth = parseFloat(itemsG[index].querySelector("rect").getAttribute("width"));
                        expect(rectWidth).toBeLessThan(rectsWidth[index]);
                    }
                })
            })
            test('GroupBy jour', async () => {
                act(() => {
                    render(<Amounts />)
                })

                await act(async () => {
                    const select = screen.getByTestId("selectGroupBy");
                    fireEvent.change(select, {target: {value: "week"}});
                })

                const g = await screen.findAllByTestId(/data:/i);
                const rectsWidth = g.map(item => parseFloat(item.querySelector("rect").getAttribute("width")))

                await act(async () => {
                    const select = screen.getByTestId("selectGroupBy");
                    fireEvent.change(select, {target: {value: "day"}});
                })

                await waitFor(() => {
                    const itemsG = screen.getAllByTestId(/data:/i)
                    for(let index in itemsG){
                        const rectWidth = parseFloat(itemsG[index].querySelector("rect").getAttribute("width"));
                        expect(rectWidth).toBeLessThan(rectsWidth[index]);
                    }
                })
            })
            test('GroupBy année', async () => {
                act(() => {
                    render(<Amounts />)
                })

                await act(async () => {
                    const select = screen.getByTestId("selectGroupBy");
                    fireEvent.change(select, {target: {value: "year"}});
                })

                await waitFor(() => {
                    const axis = screen.getAllByTestId(/xAxis:/i);
                    const transformDebut = axis[0].closest("g").getAttribute("transform")
                    const debut = parseFloat(transformDebut.substring(transformDebut.indexOf("(")+1, transformDebut.indexOf(",")))

                    const transformFin = axis[axis.length-1].closest("g").getAttribute("transform")
                    const fin = parseFloat(transformFin.substring(transformFin.indexOf("(")+1, transformFin.indexOf(",")))

                    const rect = screen.getByTestId(`data:${items[0].date}T00:00:00.000Z`).querySelector("rect");
                    expect(rect).toHaveAttribute("width", `${fin - debut}`);
                })
            })
        })
    })
})