import * as d3 from "d3";
import React, {useEffect, useState} from "react";
import {DonutChartLegend} from "../../components/d3js/Legends";
import {ApiService} from "../../components/d3js/ApiService";
import Spinner from "../../components/d3js/Spinner";
import {randomKey} from "../../components/common/randomKey";
import {resizeGraph} from "../../components/d3js/resizeGraph";

// Test items
const items = [
    {region: "Bretagne", value: 0.05},
    {region: "Nouvelle Aquitaine", value: 0.22},
    {region: "Occitanie", value: 0.16},
    {region: "Centre Val de Loire", value: 0.08},
    {region: "Normandie", value: 0.115},
    {region: "Ile de France", value: 0.13},
    {region: "Grand Est", value: 0.12},
    {region: "Autre", value: 0.125}
]
export default function Regions(){
    const [divWidth, setDivWidth] = useState(-1);
    const [divHeight, setDivHeight] = useState(-1);
    const [radius, setRadius] = useState(0);
    const [items, setItems] = useState<Region[]>([]);
    const [arcs, setArcs] = useState<Arc[]>([]);

    // Listener sur la taille de la page, récupération des données
    useEffect(() => {
        window.addEventListener("resize", () => {
            resizeGraph("divGraph", setDivWidth, setDivHeight);
        })
        ApiService.getRegions()
            .then(regions => setItems(regions));
    }, [])

    // Mis à jour du radius par rapport à la taille de la page
    useEffect(() => {
        setRadius(Math.min(divWidth, divHeight) / 2 - 50)
    }, [divWidth, divHeight])

    // Calcul des arcs par rapport aux données et au radius
    useEffect(() => {
        let start = 0;
        const regions = [...items];
        //On ajoute la région "Autre", indiquant le reste des ventes
        let total = 0;
        regions.forEach(item => total += item.percentage/100)
        if(total > 0 && total < 1)
            regions.push({region: "Autre", percentage:(1-total)*100})

        const colors = d3.scaleSequential().domain([0,regions.length])
            .interpolator(d3.interpolateViridis);

        setArcs(regions.map((item, index) => {
            const startAngle = start;
            let endAngle = startAngle + (item.percentage/100)*(Math.PI*2);
            start = endAngle;
            return {
                name: item.region,
                value: item.percentage,
                angle: {
                    startAngle,
                    endAngle,
                    innerRadius: radius * 0.5,
                    outerRadius: radius * 0.8,
                },
                color: colors(index)
            }
        }));
    }, [radius, items])

    return (
        <>
            <Spinner active={items.length <= 1} onDeactivate={() => resizeGraph("divGraph", setDivWidth, setDivHeight)} />
            <div className="d-flex justify-content-center vh-100" id="divGraph">
            {items.length > 0 &&
            <>
                <svg
                    width="100%"
                    height="100%"
                    overflow="visible"
                >
                    <DonutChart data={arcs} transform={`translate(${divWidth/2}, ${divHeight/2})`}/>
                    <DonutChartLegend data={arcs} radius={radius} transform={`translate(${divWidth/2}, ${divHeight/2})`}/>
                </svg>
            </>}
            </div>
        </>
    )
}

function DonutChart({data, transform}: {data: Arc[], transform: string}){
    const PADDING = Math.PI / 128;
    //Génère un arc par rapport à un startAngle et endAngle
    const arc = d3.arc();

    //Détermine la couleur d'un texte par rapport à la couleur du background
    const getContrast = (hexcolor: string) => {
        var r = parseInt(hexcolor.substring(1,3),16);
        var g = parseInt(hexcolor.substring(3,5),16);
        var b = parseInt(hexcolor.substring(5,7),16);
        var yiq = ((r*299)+(g*587)+(b*114))/1000;
        return (yiq >= 128) ? 'black' : 'white';
    }
    return (
        <g>
            {data.map((item) => (
                <g
                    key={randomKey()}
                    transform={transform}
                >
                    {/*Arc représentant la valeur de l'item*/}
                    <path
                        d={arc({ ...item.angle, endAngle: item.angle.endAngle - PADDING })!}
                        fill={item.color}
                    />
                    {/*Texte indiquant la valeur de l'item*/}
                    <g
                        transform={`translate(${arc.centroid(item.angle)})`}
                    >
                        <text
                            textAnchor="middle"
                            fill={getContrast(item.color)}
                        >{item.value.toFixed(2)} %</text>
                    </g>
                </g>
            ))}
        </g>
    )
}