import React from "react";
import {randomKey} from "../common/randomKey";
import * as d3 from "d3";

/**
 * Fonction générant les scales d3js pour générer les différents axes et formatter les données
 * @param items données
 * @param dateDebut date de début de l'axe abscisses
 * @param dateFin date de fin de l'axe des abscisses
 * @param xMax largeur maximale du graph
 * @param yMax hauteur maximale du graph
 */
export function getScales(items: any, dateDebut: Date, dateFin: Date, xMax: number, yMax: number):{xScale:d3.ScaleTime<number, number, never>, yScale:d3.ScaleLinear<number, number, never>}{
    const xScale = d3.scaleTime()
        .domain([dateDebut, dateFin])
        .nice()
        .range([50, xMax])

    const sortedItems = [...items]
    sortedItems.sort((a,b) => {return a.value - b.value})

    const yScale = d3.scaleLinear()
        .domain([sortedItems[sortedItems.length - 1].value, 0])
        .nice()
        .range([0, yMax])

    return {
        xScale: xScale,
        yScale: yScale
    };
}

/**
 * Fonction dessinant l'axe des abscisses
 * @param param0 
 * @returns 
 */
export function AxisX({ticks, transform}: { ticks: XTicks[], transform: string}) {
    let format: Intl.DateTimeFormatOptions = {year: "numeric"};
    if(ticks.length > 1){
        // @ts-ignore
        let diff = Math.ceil((ticks[1].value - ticks[0].value) / (1000*60*60*24));
        // Si la différence entre le premier et le 2eme tick est moins d'1 an, on affiche le nom des mois
        if(diff < 365)
            format = {month: "long", year: "numeric"};
        // Si la différence est inférieur au nombre de jours dans le mois du premier tick, on affiche le numéro du jour
        if(diff < new Date(ticks[0].value.getFullYear(), ticks[0].value.getMonth()+1, 0).getDate())
            format = {day:"numeric", month: "numeric", year: "numeric"};
    }
    return (
        <g
            key="xAxis"
            transform={transform}
        >
            {ticks.map(({ value, xOffset }) => (
                <g
                    key={randomKey()}
                    transform={`translate(${xOffset}, 0)`}
                >
                    <line
                        y2="6"
                        stroke="currentColor"
                    />
                    <text
                        key={randomKey()}
                        style={{
                            fontSize: "10px",
                            textAnchor: "middle",
                            transform: "translateY(20px)"
                        }}>
                        {new Intl.DateTimeFormat('fr-FR', format).format(value)}
                    </text>
                </g>
            ))}
            {(ticks.length>0)?(
                <g>
                    <line 
                        x1="0"
                        x2={ticks[ticks.length-1].xOffset}
                        stroke="currentColor"
                    />
                </g>
            ):<g />}
            
        </g>
    )
}

/**
 * Fonction dessinant l'axe des ordonnées
 * @param param0 
 * @returns 
 */
export function AxisY({ticks, transform, textLabel}: { ticks: YTicks[], transform: string, textLabel: JSX.Element}){
    return (
        <g
            key="yAxis"
            transform={transform}
        >
            {(ticks.length>0)?(
                <text
                    style={{
                        fontSize: "26px",
                        textAnchor: "middle",
                        transform: `translate(-95px,${ticks[ticks.length-1].yOffset/2-55}px)`
                    }}
                >
                    {textLabel}
                </text>
            ):<text />}
            {ticks.map(({ value, yOffset }) => (
                <g
                    key={randomKey()}
                    transform={`translate(-6, ${yOffset})`}
                >
                    <line
                        x2="6"
                        stroke="currentColor"
                    />
                    <text
                        key={randomKey()}
                        style={{
                            fontSize: "10px",
                            textAnchor: "end",
                            transform: "translateX(10px,20px)"
                        }}>
                        {Intl.NumberFormat('fr-FR').format(value)}
                    </text>
                </g>
            ))}
            {(ticks.length>0)?(
                <g>
                    <line 
                        y1="0"
                        y2={ticks[ticks.length-1].yOffset}
                        stroke="currentColor"
                    />
                </g>
            ):<g />}
        </g>
    )
}