
import React, { useEffect, useState, useRef } from "react";
import * as d3 from "d3";
import { nice, ticks } from "d3";

const dateFormatOptions = {month: "long", year: "numeric"}

export default function Graph2() {
    const Y_MAX = 500;
    const X_MAX = 1000;
    const MARGIN_LEFT = 50;
    const MARGIN_TOP = 50;

    const [xTicks, setXTicks] = useState<{ value: Date; xOffset: number; }[]>([])
    const [yTicks, setYTicks] = useState<{value: number, yOffset: number}[]>([])
    const [data,setData] = useState<{ date: Date; xOffset: number; amount: number, height: number}[]>([])

    useEffect(() => {
        const items = [{ date: "2017-01-01", amount: 50 }, { date: "2017-06-01", amount: 30 }, { date: "2018-01-01", amount: 70 }]

        const xScale = d3.scaleTime()
            .domain([new Date(items[0].date), new Date(items[2].date)])
            .nice()
            .range([50, X_MAX])
        
        let sortedItems = items.sort((a,b) => {return a.amount - b.amount})
        
        const yScale = d3.scaleLinear()
            .domain([sortedItems[sortedItems.length - 1].amount, 0])
            .nice()
            .range([0, Y_MAX])
        
        setXTicks(xScale.ticks()
            .map(value => ({
                value,
                xOffset: xScale(value)
            })))
        setYTicks(yScale.ticks()
            .map(value => ({
                value,
                yOffset: yScale(value)
            })))
        setData(items.map(item => {
            let date = new Date(item.date)
            let nextDate = new Date(item.date)
            nextDate.setMonth((date.getMonth()+1)%12)
            return {
                date: date, 
                xOffset: xScale(date), 
                amount: item.amount, 
                height: Y_MAX - yScale(item.amount),
                width: xScale(nextDate) - xScale(date)
            }
        }));
    },[])

    let width = 0
    if(xTicks.length > 1)
        width = xTicks[1].xOffset - xTicks[0].xOffset

    return (
        <div>
            <svg id="graph"
                width="100%"
                height={Y_MAX + MARGIN_TOP + 30}
            >
                <defs>
                    <pattern
                            id="diagonalHatch"
                            patternUnits="userSpaceOnUse" 
                            width="4" 
                            height="8" 
                            patternTransform="rotate(-45 2 2)"
                        >
                            <path d="M -1,2 l 6,0" stroke="red" stroke-width=".5"/>
                    </pattern>
                </defs>
                <AxisX ticks={xTicks} transform={`translate(${MARGIN_LEFT}, ${Y_MAX + MARGIN_TOP})`}/>
                <AxisY ticks={yTicks} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP})`} />
                <BarChart data={data} transform={`translate(${MARGIN_LEFT}, ${MARGIN_TOP*2})`} yMax={Y_MAX - MARGIN_TOP}/>
            </svg>
        </div>
    )
}  

function AxisX({ticks, transform}) {
    return (
        <g
            key="xAxis"
            transform={transform}
        >
            {ticks.map(({ value, xOffset }) => (
                <g
                    key={value.toISOString()}
                    transform={`translate(${xOffset}, 0)`}
                >
                    <line
                        y2="6"
                        stroke="currentColor"
                    />
                    <text
                        key={value.toISOString()}
                        style={{
                            fontSize: "10px",
                            textAnchor: "middle",
                            transform: "translateY(20px)"
                        }}>
                        {new Intl.DateTimeFormat('fr-FR', dateFormatOptions).format(value)}
                    </text>
                </g>
            ))}
            {(ticks.length>0)?(
                <g>
                    <line 
                        x1="0"
                        x2={ticks[ticks.length-1].xOffset}
                        stroke="currentColor"
                    />
                </g>
            ):<g />}
            
        </g>
    )
}

function AxisY({ticks, transform}){
    return (
        <g
            key="yAxis"
            transform={transform}
        >
            {ticks.map(({ value, yOffset }) => (
                <g
                    key={value}
                    transform={`translate(-6, ${yOffset})`}
                >
                    <line
                        x2="6"
                        stroke="currentColor"
                    />
                    <text
                        key={value}
                        style={{
                            fontSize: "10px",
                            textAnchor: "middle",
                            transform: "translateX(-10px)"
                        }}>
                        {value}
                    </text>
                </g>
            ))}
            {(ticks.length>0)?(
                <g>
                    <line 
                        y1="0"
                        y2={ticks[ticks.length-1].yOffset}
                        stroke="currentColor"
                    />
                </g>
            ):<g />}
        </g>
    )
}

function BarChart({data, transform, yMax}){
    //console.log(data)
    return (
        <g
            key="barChart"
            transform={transform}
        >
            {data.map(({ date, xOffset, width, height, amount }) => (
                <g
                    key={"data:"+date.toISOString()}
                    transform={`translate(${xOffset}, ${yMax - height})`}
                >
                    <rect 
                        width={width}
                        height={height}
                        stroke="red"
                        fill="url(#diagonalHatch)"
                    />
                    <text>{amount}</text>
                </g>
            ))}
        </g>
    )
}