<?php

namespace App\Controller;

use ApiPlatform\Metadata\ApiResource;
use App\Dto\BarChart\BarChartInput;
use App\Service\ChartService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Validation;
use Symfony\Component\Validator\Constraints as Assert;

class ChartController extends AbstractController
{
    private ChartService $chartService;

    public function __construct(ChartService $chartService)
    {
        $this->chartService = $chartService;
    }

    #[Route('/api/bar-chart/{startDate}/{endDate}/{granularity}', name: 'bar-chart', requirements: [
        'startDate' => '\d{4}-\d{2}-\d{2}', // YYYY-MM-DD
        'endDate' => '\d{4}-\d{2}-\d{2}',   //  YYYY-MM-DD
        'granularity' => 'day|month|year',   // day, month, ou year
    ])]
    public function getChartData(string $startDate, string $endDate, string $granularity): JsonResponse
    {
        $input = new BarChartInput();
        $input->start = $startDate;
        $input->end = $endDate;
        $input->granularity = $granularity;

        $chartData = $this->chartService->getBarChartData($input);
        $output = [];
        foreach ($chartData as $data) {
            $output[] = [
                'date' => $data->date,
                'occurrences' => $data->occurrences,
            ];
        }
        return $this->json($output);
    }

    #[Route('/api/donut-chart/{id}', name: 'donut-chart', requirements: [
        'id' => '\d{4}',
    ])]
    public function getDonutChartData(string $id): JsonResponse
    {
        $validator = Validation::createValidator();
        $violations = $validator->validate($id, [
            new Assert\NotBlank(),
            new Assert\Range(['min' => 2018, 'max' => 2023]),
        ]);

        if (count($violations) > 0) {
            return new JsonResponse(['error' => 'Not Found'], Response::HTTP_NOT_FOUND);
        }

        $data = $this->chartService->getDonutChartData($id);

        return $this->json($data);
    }
}
