<?php

namespace App\Controller;

use App\Dto\BarChartInput;
use App\Service\SaleService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Validation;
use ApiPlatform\Metadata\ApiResource;


class SaleController extends AbstractController
{
    public const GRANULARITY_DAY = 'day';
    public const GRANULARITY_MONTH = 'month';
    public const GRANULARITY_YEAR = 'year';
    private SaleService $saleService;

    public function __construct(SaleService $saleService)
    {
        $this->saleService = $saleService;
    }

    #[Route('/api/timeseries', name: 'timeseries')]
    public function getTimeSeries(): JsonResponse
    {
        $result = $this->saleService->getTimeSeries();

        return $this->json($result);
    }

    #[Route('/api/bar-chart/{startDate}/{endDate}/{granularity}', name: 'bar-chart', requirements: [
        'startDate' => '\d{4}-\d{2}-\d{2}', // YYYY-MM-DD
        'endDate' => '\d{4}-\d{2}-\d{2}',   //  YYYY-MM-DD
        'granularity' => 'day|month|year',   // day, month, ou year
    ])]
    public function getChartData(string $startDate, string $endDate, string $granularity): JsonResponse
    {
        $input = new BarChartInput();
        $input->start = $startDate;
        $input->end = $endDate;
        $input->granularity = $granularity;

        $chartData = $this->saleService->getBarChartData($input);
        $output = [];
        foreach ($chartData as $data) {
            $output[] = [
                'date' => $data->date,
                'occurrences' => $data->occurrences,
            ];
        }
        return $this->json($output);
    }

    #[Route('/api/donut-chart/{id}', name: 'donut-chart', requirements: [
        'id' => '\d{4}',
    ])]
    public function getDonutChartData(string $id): JsonResponse
    {
        $validator = Validation::createValidator();
        $violations = $validator->validate($id, [
            new Assert\NotBlank(),
            new Assert\Range(['min' => 2018, 'max' => 2023]),
        ]);

        if (count($violations) > 0) {
            return new JsonResponse(['error' => 'Not Found'], Response::HTTP_NOT_FOUND);
        }

        $data = $this->saleService->getDonutChartData($id);

        return $this->json($data);
    }
}
