<?php

namespace App\Service;

use App\Dto\BarChart\BarChartInput;
use App\Dto\BarChart\BarChartOutput;
use Doctrine\ORM\EntityManagerInterface;
use App\Entity\Sale;

class BarChartService
{
    private EntityManagerInterface $entityManager;

    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    public function getBarChartData(BarChartInput $input): array
    {
        $startDate = new \DateTime($input->start);
        $endDate = new \DateTime($input->end);

        $queryBuilder = $this->entityManager->createQueryBuilder();
        $output = [];

        if ($input->granularity == 'month') {
            $result = $queryBuilder
                ->select('YEAR(s.date) as year', 'MONTH(s.date) as month', 'COUNT(s.id) as occurrences')
                ->from(Sale::class, 's')
                ->where('s.date BETWEEN :start AND :end')
                ->setParameter('start', $startDate)
                ->setParameter('end', $endDate)
                ->groupBy('year', 'month')
                ->orderBy('year')
                ->getQuery()
                ->getResult();
        } else {
            switch ($input->granularity) {
                case 'day':
                    $groupByExpression = 's.date';
                    $groupByAlias = 'date';
                    $dateFormat = 'Y-m-d';
                    break;
                case 'year':
                    $groupByExpression = 'YEAR(s.date)';
                    $groupByAlias = 'year';
                    $dateFormat = 'Y';
                    break;
                default:
                    throw new \InvalidArgumentException('Invalid granularity');
            }

            $result = $queryBuilder
                ->select("{$groupByExpression} as {$groupByAlias}", 'COUNT(s.id) as occurrences')
                ->from(Sale::class, 's')
                ->where('s.date BETWEEN :start AND :end')
                ->setParameter('start', $startDate)
                ->setParameter('end', $endDate)
                ->groupBy("{$groupByAlias}")
                ->orderBy("{$groupByAlias}")
                ->getQuery()
                ->getResult();
        }

        foreach ($result as $row) {
            if ($input->granularity == 'month') {
                $month = $row['month'];
                $year = $row['year'];
                $dateString = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT);
            } else {
                $dateString = $row[$groupByAlias] instanceof \DateTimeInterface ? $row[$groupByAlias]->format($dateFormat) : $row[$groupByAlias];
            }

            $output[] = new BarChartOutput($dateString, (int)$row['occurrences']);
        }

        return $output;
    }
}

