package collection;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Stack;

/**
 * Arbre Rouge-Noir générique avec sentinelle (NIL) pour représenter les feuilles nulles.
 * La sentinelle est un nœud unique et immuable de couleur NOIR.
 */
public class RedBlackTree<E extends Comparable<E>> implements Collection<E> {
  static final boolean ROUGE = true;
  static final boolean NOIR = false;

  /** Classe interne représentant un nœud de l'arbre */
  class Noeud {
    E cle;
    Noeud gauche;
    Noeud droit;
    Noeud parent;
    boolean color = ROUGE; 

    /** Constructeur pour la Sentinelle NIL (toujours NOIR) */
    private Noeud() {
        this.color = NOIR; 
        this.cle = null;   
        // Les liens de la sentinelle peuvent pointer vers elle-même ou être nulls, 
        // mais pour la simplicité, nous initialisons les autres champs de données plus bas.
    }

    /** Constructeur pour un nouveau nœud de données (par défaut ROUGE) */
    Noeud(E cle, Noeud parent) {
      this.cle = cle;
      this.parent = parent;
      this.gauche = NIL; // Les nouvelles feuilles pointent vers NIL
      this.droit = NIL;
    }
  }

  // --- Initialisation de la Sentinelle et de la Racine ---
  
  // La sentinelle NIL est un nœud noir sans données
  private final Noeud NIL = new Noeud(); 
  
  private Noeud racine = NIL; // La racine est initialisée à la sentinelle quand l'arbre est vide
  private int taille = 0;

  // --- Méthodes de Rééquilibrage ---

  /** Rotation gauche standard */
  private void rotateLeft(Noeud x) {
    Noeud y = x.droit;
    x.droit = y.gauche;

    // Mise à jour du parent de l'enfant gauche de y
    y.gauche.parent = x;

    y.parent = x.parent;

    if (x.parent == NIL) { // Test par rapport à NIL, pas null
      racine = y;
    } else if (x == x.parent.gauche) {
      x.parent.gauche = y;
    } else {
      x.parent.droit = y;
    }

    y.gauche = x;
    x.parent = y;
  }

  /** Rotation droite standard */
  private void rotateRight(Noeud x) {
    Noeud y = x.gauche;
    x.gauche = y.droit;

    // Mise à jour du parent de l'enfant droit de y
    y.droit.parent = x;

    y.parent = x.parent;

    if (x.parent == NIL) { // Test par rapport à NIL, pas null
      racine = y;
    } else if (x == x.parent.droit) {
      x.parent.droit = y;
    } else {
      x.parent.gauche = y;
    }

    y.droit = x;
    x.parent = y;
  }
  
  

  /**
   * Correction après insertion. Assure que les propriétés RB sont conservées.
   * Simplifiée : L'oncle 'y' est toujours un objet (soit un noeud ROUGE, soit NIL-NOIR).
   */
  private void fixAfterInsert(Noeud z) {
    while (z.parent.color == ROUGE) { // z.parent n'est jamais NIL ici
      if (z.parent == z.parent.parent.gauche) {
        Noeud y = z.parent.parent.droit; // oncle

        if (y.color == ROUGE) {
          // Cas 1 : parent rouge + oncle rouge -> recolorations
          z.parent.color = NOIR;
          y.color = NOIR;
          z.parent.parent.color = ROUGE;
          z = z.parent.parent;
        } else {
          if (z == z.parent.droit) {
            // Cas 2 : triangle -> rotation gauche
            z = z.parent;
            rotateLeft(z);
          }

          // Cas 3 : ligne -> rotation droite
          z.parent.color = NOIR;
          z.parent.parent.color = ROUGE;
          rotateRight(z.parent.parent);
        }
      } else {
        // Symétrique : parent est enfant droit
        Noeud y = z.parent.parent.gauche;

        if (y.color == ROUGE) {
          z.parent.color = NOIR;
          y.color = NOIR;
          z.parent.parent.color = ROUGE;
          z = z.parent.parent;
        } else {
          if (z == z.parent.gauche) {
            z = z.parent;
            rotateRight(z);
          }
          z.parent.color = NOIR;
          z.parent.parent.color = ROUGE;
          rotateLeft(z.parent.parent);
        }
      }
    }

    racine.color = NOIR;
  }

  /** Remplace un nœud par un autre dans l’arbre */
  private void transplant(Noeud u, Noeud v) {
    if (u.parent == NIL) {
      racine = v;
    } else if (u == u.parent.gauche) {
      u.parent.gauche = v;
    } else {
      u.parent.droit = v;
    }

    // V est soit un nœud de données, soit NIL.
    v.parent = u.parent;
  }

  /** Renvoie le minimum du sous-arbre */
  private Noeud minimum(Noeud x) {
    while (x.gauche != NIL) { // Test par rapport à NIL
      x = x.gauche;
    }

    return x;
  }

  /**
   * Correction après suppression. Restaure les propriétés Rouge-Noir.
   * Simplifiée : x est toujours un objet (soit un nœud de données, soit NIL).
   */
  private void fixAfterDelete(Noeud x, Noeud parent) {
    // La condition inclut 'x != NIL' au lieu de 'x != null' 
    while ((x != racine) && (x.color == NOIR)) {
      if (x == parent.gauche) {
        Noeud w = parent.droit; // w ne peut pas être NIL ici, car sinon cela violerait la propriété RB

        if (w.color == ROUGE) {
          // Cas 1 : frère (w) est ROUGE
          w.color = NOIR;
          parent.color = ROUGE;
          rotateLeft(parent);
          w = parent.droit; // Nouveau frère
        }

        // Cas 2 : frère (w) NOIR, ses deux enfants NIL (NOIR)
        if (w.gauche.color == NOIR && w.droit.color == NOIR) {
          w.color = ROUGE;
          x = parent;
          parent = x.parent;
        } else {
          // Cas 3 : frère (w) NOIR, enfant Droit NOIR, enfant Gauche ROUGE
          if (w.droit.color == NOIR) {
            w.gauche.color = NOIR;
            w.color = ROUGE;
            rotateRight(w);
            w = parent.droit; // Nouveau frère
          }

          // Cas 4 : frère (w) NOIR, enfant Droit ROUGE
          w.color = parent.color;
          parent.color = NOIR;
          w.droit.color = NOIR;
          rotateLeft(parent);
          x = racine; // Terminer la boucle
          break;
        }
      } else {
        // Symétrique : x est enfant droit
        Noeud w = parent.gauche;

        if (w.color == ROUGE) {
          w.color = NOIR;
          parent.color = ROUGE;
          rotateRight(parent);
          w = parent.gauche;
        }

        if (w.droit.color == NOIR && w.gauche.color == NOIR) {
          w.color = ROUGE;
          x = parent;
          parent = x.parent;
        } else {
          if (w.gauche.color == NOIR) {
            w.droit.color = NOIR;
            w.color = ROUGE;
            rotateLeft(w);
            w = parent.gauche;
          }

          w.color = parent.color;
          parent.color = NOIR;
          w.gauche.color = NOIR;
          rotateRight(parent);
          x = racine;
          break;
        }
      }
    }
    
    // Si x est NIL à la fin, sa couleur est déjà NOIR.
    if (x != NIL) {
      x.color = NOIR;
    }
  }

  // --- Implémentation de Collection ---

  @Override
  public boolean add(E e) {
    Objects.requireNonNull(e);

    if (racine == NIL) {
      racine = new Noeud(e, NIL);
      racine.color = NOIR;
      taille = 1;
      return true;
    }

    Noeud cur = racine;
    Noeud parent = NIL;
    int cmp = 0;

    // Recherche de la position d'insertion
    while (cur != NIL) {
      parent = cur;
      cmp = e.compareTo(cur.cle);

      if (cmp < 0) {
        cur = cur.gauche;
      } else if (cmp > 0) {
        cur = cur.droit;
      } else {
        return false; // pas de doublons
      }
    }

    // Le constructeur du nœud initialise gauche/droit à NIL
    Noeud node = new Noeud(e, parent); 

    if (cmp < 0) {
      parent.gauche = node;
    } else {
      parent.droit = node;
    }

    fixAfterInsert(node);
    taille++;

    return true;
  }

  /** Recherche un nœud contenant la clé */
  private Noeud findNode(E e) {
    Noeud cur = racine;

    while (cur != NIL) {
      int cmp = e.compareTo(cur.cle);

      if (cmp == 0) {
        return cur;
      }
      if (cmp < 0) {
        cur = cur.gauche;
      } else {
        cur = cur.droit;
      }
    }

    return NIL; // Retourne NIL si non trouvé, pas null
  }

  @Override
  public boolean contains(Object o) {
    if (o == null) {
      return false;
    }

    try {
      @SuppressWarnings("unchecked")
      E e = (E) o;
      return findNode(e) != NIL;
    } catch (ClassCastException ex) {
      return false;
    }
  }

  @Override
  public boolean remove(Object o) {
    // ... (vérification de type)

    E e;
    try {
      e = (E) o;
    } catch (ClassCastException ex) {
      return false;
    }

    Noeud z = findNode(e);

    if (z == NIL) {
      return false;
    }

    Noeud y = z;
    boolean yorigColor = y.color;
    Noeud x; // Le nœud de remplacement (peut être NIL)
    Noeud xparent;

    // Cas 1 : un seul enfant (ou aucun)
    if (z.gauche == NIL) {
      x = z.droit;
      xparent = z.parent;
      transplant(z, z.droit);
    } else if (z.droit == NIL) {
      x = z.gauche;
      xparent = z.parent;
      transplant(z, z.gauche);
    } else { // Cas 2 : deux enfants → successeur
      y = minimum(z.droit);
      yorigColor = y.color;
      x = y.droit; // x est soit un nœud de données, soit NIL

      if (y.parent == z) {
        // x.parent est toujours settable
        x.parent = y;
        xparent = y;
      } else {
        transplant(y, y.droit); // y.droit peut être NIL
        y.droit = z.droit;

        y.droit.parent = y;

        xparent = y.parent;
      }

      transplant(z, y);
      y.gauche = z.gauche;

      y.gauche.parent = y;

      y.color = z.color;
    }

    if (yorigColor == NOIR) {
      fixAfterDelete(x, xparent);
    }

    taille--;
    return true;
  }

  @Override
  public Iterator<E> iterator() {
    return new Iterator<E>() {
      private final Stack<Noeud> stack = initStack(racine);

      private Stack<Noeud> initStack(Noeud r) {
        Stack<Noeud> s = new Stack<>();
        Noeud cur = r;

        while (cur != NIL) { // Test par rapport à NIL
          s.push(cur);
          cur = cur.gauche;
        }

        return s;
      }

      @Override
      public boolean hasNext() {
        return !stack.isEmpty();
      }

      @Override
      public E next() {
        if (!hasNext()) {
          throw new NoSuchElementException();
        }

        Noeud n = stack.pop();
        E res = n.cle;
        Noeud cur = n.droit;

        // Descente maximale à gauche
        while (cur != NIL) { // Test par rapport à NIL
          stack.push(cur);
          cur = cur.gauche;
        }

        return res;
      }

      @Override
      public void remove() {
        throw new UnsupportedOperationException();
      }
    };
  }

  // --- Méthodes restantes (Identiques) ---

  @Override
  public int size() {
    return taille;
  }

  @Override
  public boolean isEmpty() {
    return taille == 0;
  }

  @Override
  public void clear() {
    racine = NIL; // La racine redevient la sentinelle
    taille = 0;
  }

  @Override
  public Object[] toArray() {
    Object[] arr = new Object[taille];
    int i = 0;

    for (E e : this) {
      arr[i++] = e;
    }

    return arr;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> T[] toArray(T[] a) {
    if (a.length < taille) {
      a = (T[]) java.lang.reflect.Array.newInstance(a.getClass().getComponentType(), taille);
    }

    int i = 0;

    for (E e : this) {
      a[i++] = (T) e;
    }

    if (a.length > taille) {
      a[taille] = null;
    }

    return a;
  }

  @Override
  public boolean containsAll(Collection<?> c) {
    for (Object o : c) {
      if (!contains(o)) {
        return false;
      }
    }

    return true;
  }

  @Override
  public boolean addAll(Collection<? extends E> c) {
    boolean changed = false;
    for (E e : c) {
      changed |= add(e);
    }

    return changed;
  }

  @Override
  public boolean removeAll(Collection<?> c) {
    boolean changed = false;
    for (Object o : c) {
      changed |= remove(o);
    }

    return changed;
  }

  @Override
  public boolean retainAll(Collection<?> c) {
    List<E> toRemove = new ArrayList<>();

    for (E e : this) {
      if (!c.contains(e)) {
        toRemove.add(e);
      }
    }

    for (E e : toRemove) {
      remove(e);
    }

    return !toRemove.isEmpty();
  }
}