package collection;

import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Random;

public class Benchmark {

  private static final Random RNG = new Random(12345);

  public static void main(final String[] args) throws Exception {

    final int[] nList = { 1000, 2000, 5000, 10000, 20000, 50000 };
    final int repetition = 5;

    // --- Creation des 4 CSV avec leur enTete ---
    PrintWriter abrInsert = new PrintWriter(new FileWriter("ABR_insert.csv"));
    writeHeader(abrInsert, nList);

    PrintWriter abrSearch = new PrintWriter(new FileWriter("ABR_search.csv"));
    writeHeader(abrSearch, nList);

    PrintWriter arnInsert = new PrintWriter(new FileWriter("ARN_insert.csv"));
    writeHeader(arnInsert, nList);

    PrintWriter arnSearch = new PrintWriter(new FileWriter("ARN_search.csv"));
    writeHeader(arnSearch, nList);

    // Scenarios
    List<String> scenarios = Arrays.asList("ASCENDANT", "ALEATOIRE");

    // Process each structure
    for (String structure : Arrays.asList("ABR", "ARN")) {

      for (String scenario : scenarios) {

        // Buffers for one line of insert & search results (Maintenant List<Long>)
        List<Long> insertLine = new ArrayList<>();
        List<Long> searchLine = new ArrayList<>();

        for (int n : nList) {

          long insertSum = 0;
          long searchSum = 0;

          for (int r = 0; r < repetition; r++) {

            List<Integer> keys = new ArrayList<>();
            for (int i = 0; i < n; i++) {
              keys.add(i);
            }

            if (scenario.equals("ALEATOIRE")) {
              Collections.shuffle(keys, new Random(RNG.nextLong()));
            }

            long[] result = structure.equals("ABR") ? benchmarkAbr(keys, n) : benchmarkArn(keys, n);

            insertSum += result[0];
            searchSum += result[1];
          }

          // Calcul de la moyenne en utilisant la division entière
          insertLine.add(insertSum / repetition);
          searchLine.add(searchSum / repetition);

          System.out.printf("Done %s %s n=%d\n", structure, scenario, n);
        }

        // Write to correct CSV
        if (structure.equals("ABR")) {
          writeScenarioLine(abrInsert, scenario, insertLine);
          writeScenarioLine(abrSearch, scenario, searchLine);
        } else {
          writeScenarioLine(arnInsert, scenario, insertLine);
          writeScenarioLine(arnSearch, scenario, searchLine);
        }
      }
    }

    abrInsert.close();
    abrSearch.close();
    arnInsert.close();
    arnSearch.close();

    System.out.println("4 CSV créés au format tableau avec des entiers.");
  }

  // -------------------------------------------------------
  // Helper: header line
  // -------------------------------------------------------
  private static void writeHeader(PrintWriter pw, int[] nlist) {
    pw.print("n");
    for (int n : nlist) {
      pw.print(";" + n);
    }

    pw.println();
  }

  // -------------------------------------------------------
  // Helper: write one scenario line (Modification pour gérer List<Long>)
  // -------------------------------------------------------
  private static void writeScenarioLine(PrintWriter pw, String scenario, List<Long> values) {
    pw.print(scenario);
    for (long v : values) {
      // Utilisation de %d pour garantir l'affichage d'un entier
      pw.printf(";%d", v); 
    }

    pw.println();
  }

  // -------------------------------------------------------
  // Benchmark ABR / ARN
  // -------------------------------------------------------

  private static long[] benchmarkAbr(List<Integer> keys, int n) {
    Collection<Integer> tree = new ArbreBinaireRecherche<>();
    return runBenchmark(tree, keys, n);
  }

  private static long[] benchmarkArn(List<Integer> keys, int n) {
    // Assurez-vous d'avoir une classe RedBlackTree dans le package collection 
    // qui implémente Collection<Integer> pour que ce code compile.
    Collection<Integer> tree = new RedBlackTree<>(); 
    return runBenchmark(tree, keys, n);
  }

  private static long[] runBenchmark(Collection<Integer> tree, List<Integer> keys, int n) {
    // Mesure du temps d'insertion
    long t0 = System.nanoTime();
    for (Integer k : keys) {
      tree.add(k);
    }
    long t1 = System.nanoTime();

    // Mesure du temps de recherche (contains)
    long t2 = System.nanoTime();
    for (int x = 0; x < 2 * n; x++) {
      tree.contains(x);
    }
    long t3 = System.nanoTime();

    // Retourne [temps insertion, temps recherche] en nanosecondes (long)
    return new long[] { t1 - t0, t3 - t2 };
  }
}