package collection;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class RedBlackTreeTest {

  private RedBlackTree<Integer> tree;

  @BeforeEach
  void setUp() {
    tree = new RedBlackTree<>();
  }

  @Test
  void testAddAndContains() {
    assertTrue(tree.add(10));
    assertTrue(tree.contains(10));
    assertFalse(tree.add(10)); // pas de doublons
  }

  @Test
  void testRemove() {
    tree.add(20);
    tree.add(10);
    tree.add(30);

    assertTrue(tree.remove(20));
    assertFalse(tree.contains(20));
    assertFalse(tree.remove(40)); // élément inexistant
  }

  @Test
  void testSizeAndIsEmpty() {
    assertTrue(tree.isEmpty());
    tree.add(1);
    tree.add(2);
    tree.add(3);

    assertEquals(3, tree.size());
    assertFalse(tree.isEmpty());
  }

  @Test
  void testClear() {
    tree.add(5);
    tree.add(15);
    tree.clear();
    assertEquals(0, tree.size());
    assertTrue(tree.isEmpty());
  }

  @Test
  void testIterator() {
    tree.add(3);
    tree.add(1);
    tree.add(4);
    tree.add(2);

    Iterator<Integer> it = tree.iterator();
    int[] expectedOrder = { 1, 2, 3, 4 };
    int i = 0;

    while (it.hasNext()) {
      assertEquals(expectedOrder[i++], it.next());
    }

    assertEquals(expectedOrder.length, i);
  }

  @Test
  void testToArray() {
    tree.add(7);
    tree.add(3);
    tree.add(9);

    Object[] arr = tree.toArray();
    Arrays.sort(arr); // pour comparaison
    assertArrayEquals(new Object[] { 3, 7, 9 }, arr);
  }

  @Test
  void testToArrayWithType() {
    tree.add(5);
    tree.add(1);
    tree.add(8);

    Integer[] arr = new Integer[3];
    arr = tree.toArray(arr);

    Arrays.sort(arr);
    assertArrayEquals(new Integer[] { 1, 5, 8 }, arr);
  }

  @Test
  void testAddAllAndContainsAll() {
    List<Integer> list = Arrays.asList(10, 20, 30);
    assertTrue(tree.addAll(list));
    assertTrue(tree.containsAll(list));
  }

  @Test
  void testRemoveAll() {
    tree.add(1);
    tree.add(2);
    tree.add(3);

    List<Integer> toRemove = Arrays.asList(2, 3, 4);
    assertTrue(tree.removeAll(toRemove));
    assertFalse(tree.contains(2));
    assertFalse(tree.contains(3));
    assertTrue(tree.contains(1));
  }

  @Test
  void testRetainAll() {
    tree.add(1);
    tree.add(2);
    tree.add(3);

    List<Integer> toRetain = Arrays.asList(2, 3, 4);
    assertTrue(tree.retainAll(toRetain));
    assertFalse(tree.contains(1));
    assertTrue(tree.contains(2));
    assertTrue(tree.contains(3));
  }
}
