package collection;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class RedBlackTreeTest {

  private RedBlackTree<Integer> tree;

  @BeforeEach
  void setUp() {
    tree = new RedBlackTree<>();
  }

  @Test
  void testAddAndContains() {
    assertTrue(tree.add(10));
    assertTrue(tree.contains(10));
    assertFalse(tree.add(10)); // doublon
  }

  @Test
  void testAddNullThrowsException() {
    assertThrows(NullPointerException.class, () -> tree.add(null));
  }

  @Test
  void testContainsWithInvalidType() {
    assertFalse(tree.contains("string"));
    assertFalse(tree.contains(null));
  }

  @Test
  void testRemove() {
    tree.add(20);
    tree.add(10);
    tree.add(30);

    // suppression racine (cas avec 2 enfants)
    assertTrue(tree.remove(20));
    assertFalse(tree.contains(20));

    // suppression feuille
    assertTrue(tree.remove(10));
    assertFalse(tree.contains(10));

    // suppression élément inexistant
    assertFalse(tree.remove(40));
  }

  @Test
  void testSizeAndIsEmpty() {
    assertTrue(tree.isEmpty());
    tree.add(1);
    tree.add(2);
    tree.add(3);

    assertEquals(3, tree.size());
    assertFalse(tree.isEmpty());
  }

  @Test
  void testClear() {
    tree.add(5);
    tree.add(15);
    tree.clear();
    assertEquals(0, tree.size());
    assertTrue(tree.isEmpty());
  }

  @Test
  void testIteratorOrder() {
    tree.add(3);
    tree.add(1);
    tree.add(4);
    tree.add(2);

    Iterator<Integer> it = tree.iterator();
    int[] expectedOrder = { 1, 2, 3, 4 };
    int i = 0;

    while (it.hasNext()) {
      assertEquals(expectedOrder[i++], it.next());
    }

    assertEquals(expectedOrder.length, i);
  }

  @Test
  void testIteratorNextException() {
    Iterator<Integer> it = tree.iterator();
    assertThrows(NoSuchElementException.class, it::next);
  }

  @Test
  void testIteratorRemoveException() {
    tree.add(1);
    Iterator<Integer> it = tree.iterator();
    assertThrows(UnsupportedOperationException.class, it::remove);
  }

  @Test
  void testToArray() {
    tree.add(7);
    tree.add(3);
    tree.add(9);

    Object[] arr = tree.toArray();
    Arrays.sort(arr);
    assertArrayEquals(new Object[] { 3, 7, 9 }, arr);
  }

  @Test
  void testToArrayWithType() {
    tree.add(5);
    tree.add(1);
    tree.add(8);

    Integer[] arr = new Integer[2]; // tableau trop petit
    arr = tree.toArray(arr);

    Arrays.sort(arr);
    assertArrayEquals(new Integer[] { 1, 5, 8 }, arr);

    Integer[] bigger = new Integer[5];
    arr = tree.toArray(bigger);
    Arrays.sort(Arrays.copyOf(arr, tree.size()));
    assertArrayEquals(new Integer[] { 1, 5, 8 }, Arrays.copyOf(arr, tree.size()));
  }

  @Test
  void testAddAllAndContainsAll() {
    List<Integer> list = Arrays.asList(10, 20, 30);
    assertTrue(tree.addAll(list));
    assertTrue(tree.containsAll(list));

    // ajout vide
    assertFalse(tree.addAll(Arrays.asList()));
    assertTrue(tree.containsAll(Arrays.asList()));
  }

  @Test
  void testRemoveAll() {
    tree.add(1);
    tree.add(2);
    tree.add(3);

    List<Integer> toRemove = Arrays.asList(2, 3, 4);
    assertTrue(tree.removeAll(toRemove));
    assertFalse(tree.contains(2));
    assertFalse(tree.contains(3));
    assertTrue(tree.contains(1));

    // suppression vide
    assertFalse(tree.removeAll(Arrays.asList()));
  }

  @Test
  void testRetainAll() {
    tree.add(1);
    tree.add(2);
    tree.add(3);

    List<Integer> toRetain = Arrays.asList(2, 3, 4);
    assertTrue(tree.retainAll(toRetain));
    assertFalse(tree.contains(1));
    assertTrue(tree.contains(2));
    assertTrue(tree.contains(3));

    // conserver tout
    assertFalse(tree.retainAll(Arrays.asList(2, 3)));
  }

  @Test
  void testEdgeCases() {
    // arbre vide
    assertEquals(0, tree.size());
    assertTrue(tree.isEmpty());
    assertFalse(tree.contains(5));
    assertFalse(tree.remove(5));

    // ajouter un seul élément
    tree.add(42);
    assertEquals(1, tree.size());
    assertTrue(tree.contains(42));
  }

  @Test
  void testFixAfterDeletePaths() {
    // Création d'un arbre complexe pour couvrir toutes les branches
    tree.add(20);
    tree.add(10);
    tree.add(30);
    tree.add(5);
    tree.add(15);
    tree.add(25);
    tree.add(35);

    // Suppression d'un nœud noir avec frère noir et enfant rouge
    tree.remove(5); // feuille noire, parent a frère noir
    tree.remove(15); // nœud noir avec frère rouge
    tree.remove(25); // nœud noir avec frère noir
    tree.remove(30); // déclenche rotation gauche dans fixAfterDelete

    // Vérification que l'arbre reste cohérent
    assertFalse(tree.contains(5));
    assertFalse(tree.contains(15));
    assertFalse(tree.contains(25));
    assertFalse(tree.contains(30));

    // Suppression racine
    tree.remove(20);
    assertFalse(tree.contains(20));
  }

  @Test
  void testFixAfterDeleteRedAndBlackSibling() {
    // Construction de l'arbre pour tester tous les cas
    tree.add(20); // racine noire
    tree.add(10); // noir
    tree.add(30); // noir
    tree.add(5); // rouge
    tree.add(15); // rouge
    tree.add(25); // rouge
    tree.add(35); // rouge

    // Supprimer des feuilles noires pour déclencher fixAfterDelete
    tree.remove(10); // x = 10 (noir), frère 30 (noir) → condition w non null, w.color NOIR
    tree.remove(5); // x = 5 (rouge ou noir selon recolor) → teste plusieurs branches
    tree.remove(30); // x = 30, frère 25 (rouge) → branche frère rouge
  }

  @Test
  void testFixAfterDeleteAllPaths() {
    tree.add(20); // racine noire
    tree.add(10);
    tree.add(30);
    tree.add(5);
    tree.add(15);
    tree.add(25);
    tree.add(35);

    tree.remove(10);
    assertFalse(tree.contains(10)); // vérifie suppression

    tree.remove(5);
    assertFalse(tree.contains(5));

    tree.remove(25);
    assertFalse(tree.contains(25));

    tree.remove(20);
    assertFalse(tree.contains(20));

    tree.remove(15);
    assertFalse(tree.contains(15));
  }
}
