<?php
require_once "pieceSquadroUI.php";
require_once "ArrayPieceSquadro.php";
require_once "ActionSquadro.php";
require_once "PieceSquadro.php";
require_once "SquadroUIGenerator.php";
/** 
 * Classe représentant le plateau de jeu Squadro.
 */
class PlateauSquadro {
    // Vitesses des pièces selon leur trajet
    public const BLANC_V_ALLER = [0, 1, 3, 2, 3, 1, 0];
    public const BLANC_V_RETOUR = [0, 3, 1, 2, 1, 3, 0];
    public const NOIR_V_ALLER = [0, 3, 1, 2, 1, 3, 0];
    public const NOIR_V_RETOUR = [0, 1, 3, 2, 3, 1, 0];

    /** @var array Plateau de jeu (tableau de 7 lignes d'ArrayPieceSquadro) */
    private array $plateau;

    /** @var array Indices des lignes jouables */
    private array $lignesJouables = [1, 2, 3, 4, 5];

    /** @var array Indices des colonnes jouables */
    private array $colonnesJouables = [1, 2, 3, 4, 5];

    /**
     * Constructeur initialisant le plateau de jeu.
     */
    public function __construct() {
        $this->initPlateau();
    }

    /**
     * Initialise le plateau avec les pièces et cases nécessaires.
     */
    private function initPlateau(): void {
        $this->plateau = array_fill(0, 7, new ArrayPieceSquadro());

        for ($y = 0; $y < 7; $y++) {
            for ($x = 0; $x < 7; $x++) {
                if (in_array($x, [0, 6]) && in_array($y, [0, 6])) {
                    $this->plateau[$y]->add(PieceSquadro::initVide());
                } elseif ($x == 0) {
                    $this->plateau[$y]->add(PieceSquadro::initBlancOuest());
                } elseif ($x == 6) {
                    $this->plateau[$y]->add(PieceSquadro::initBlancEst());
                } elseif ($y == 0) {
                    $this->plateau[$y]->add(PieceSquadro::initNoirNord());
                } elseif ($y == 6) {
                    $this->plateau[$y]->add(PieceSquadro::initNoirSud());
                } else {
                    $this->plateau[$y]->add(PieceSquadro::initVide());
                }
            }
        }
    }

    /**
     * Retourne le plateau sous forme de tableau.
     * 
     * @return array
     */
    public function getPlateau(): array {
        return $this->plateau;
    }

    /**
     * Retire une ligne jouable.
     * 
     * @param int $index Index de la ligne à retirer
     */
    public function retireLigneJouable(int $index): void {
        $this->lignesJouables = array_diff($this->lignesJouables, [$index]);
    }

    /**
     * Retire une colonne jouable.
     * 
     * @param int $index Index de la colonne à retirer
     */
    public function retireColonneJouable(int $index): void {
        $this->colonnesJouables = array_diff($this->colonnesJouables, [$index]);
    }

    /**
     * Convertit le plateau en JSON.
     * 
     * @return string
     */
    public function toJson(): string {
        return json_encode($this->plateau);
    }

    /**
     * Crée une instance de PlateauSquadro à partir d'une chaîne JSON.
     * 
     * @param string $json Représentation JSON du plateau
     * @return PlateauSquadro
     */
    public static function fromJson(string $json): PlateauSquadro {
        $data = json_decode($json, true);
        $plateau = new PlateauSquadro();
        $plateau->plateau = $data;
        return $plateau;
    }

    public function checkVictoire($joueur) {
        // Supposons que la victoire se base sur le fait qu'un joueur ait déplacé toutes ses pièces dans une zone spécifique du plateau
        // Ici on vérifie si toutes les pièces du joueur sont sur une ligne ou une colonne finale du plateau
        foreach ($this->plateau as $x => $row) {
            foreach ($row as $y => $case) {
                if ($case->getPiece() && $case->getPiece()->getProprietaire() == $joueur) {
                    // Condition de victoire : exemple simple où les pièces doivent être dans les cases de la dernière ligne
                    if ($x == 7) { // Si la pièce se trouve dans la dernière ligne (exemple)
                        return true; // Victoire !
                    }
                }
            }
        }
        return false; // Pas encore de victoire
    }
}
?>


