<?php
require_once "pieceSquadroUI.php";
require_once "arrayPieceSquadro.php";
require_once "PlateauSquadro.php";
require_once "ActionSquadro.php";
require_once "SquadroUIGenerator.php";
/**
 * Classe représentant une pièce du jeu Squadro.
 */
class PieceSquadro {
    // Constantes pour les couleurs et types de cases
    public const BLANC = 0;
    public const NOIR = 1;
    public const VIDE = -1;
    public const NEUTRE = -2;

    // Constantes pour les directions
    public const NORD = 0;
    public const EST = 1;
    public const SUD = 2;
    public const OUEST = 3;

    /** @var int Couleur de la pièce */
    private int $couleur;

    /** @var int Direction actuelle de la pièce */
    private int $direction;

    /**
     * Constructeur privé pour imposer l'utilisation des méthodes statiques.
     * 
     * @param int $couleur Couleur de la pièce
     * @param int $direction Direction de déplacement
     */
    private function __construct(int $couleur, int $direction) {
        $this->couleur = $couleur;
        $this->direction = $direction;
    }

    /**
     * Retourne la couleur de la pièce.
     * 
     * @return int
     */
    public function getCouleur(): int {
        return $this->couleur;
    }

    /**
     * Retourne la direction de la pièce.
     * 
     * @return int
     */
    public function getDirection(): int {
        return $this->direction;
    }

    /**
     * Inverse la direction de la pièce (utile lorsqu'elle atteint un point de retournement).
     */
    public function inverseDirection(): void {
        $this->direction = ($this->direction + 2) % 4;
    }

    /**
     * Initialise une case vide.
     * 
     * @return PieceSquadro
     */
    public static function initVide(): PieceSquadro {
        return new PieceSquadro(self::VIDE, self::NORD);
    }

    /**
     * Initialise une pièce noire qui commence au nord.
     * 
     * @return PieceSquadro
     */
    public static function initNoirNord(): PieceSquadro {
        return new PieceSquadro(self::NOIR, self::NORD);
    }

    /**
     * Initialise une pièce noire qui commence au sud.
     * 
     * @return PieceSquadro
     */
    public static function initNoirSud(): PieceSquadro {
        return new PieceSquadro(self::NOIR, self::SUD);
    }

    /**
     * Initialise une pièce blanche qui commence à l'est.
     * 
     * @return PieceSquadro
     */
    public static function initBlancEst(): PieceSquadro {
        return new PieceSquadro(self::BLANC, self::EST);
    }

    /**
     * Initialise une pièce blanche qui commence à l'ouest.
     * 
     * @return PieceSquadro
     */
    public static function initBlancOuest(): PieceSquadro {
        return new PieceSquadro(self::BLANC, self::OUEST);
    }

    /**
     * Convertit la pièce en une représentation textuelle.
     * 
     * @return string
     */
    public function __toString(): string {
        return "PieceSquadro(couleur={$this->couleur}, direction={$this->direction})";
    }

    /**
     * Convertit la pièce en format JSON.
     * 
     * @return string
     */
    public function toJson(): string {
        return json_encode([
            'couleur' => $this->couleur,
            'direction' => $this->direction
        ]);
    }

    /**
     * Crée une instance de PieceSquadro à partir d'une chaîne JSON.
     * 
     * @param string $json Représentation JSON de la pièce
     * @return PieceSquadro
     */
    public static function fromJson(string $json): PieceSquadro {
        $data = json_decode($json, true);
        return new PieceSquadro($data['couleur'], $data['direction']);
    }
}
