#include "generator.h"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define BUFFER_SIZE 512

/**
 * GESTION DE LA COMMUNICATION PAR SOCKET
 * Cette fonction agit soit comme un serveur (réception et affichage),
 * soit comme un client (envoi des statistiques locales).
 */
int communicate_with_socket(int is_server, double t_r, double t_r48, double sd_r, double sd_r48, double deb_r, double deb_r48, size_t max_r, size_t min_r) {
    int sockfd = -1, newsockfd = -1;
    struct sockaddr_in serv_addr, cli_addr;
    socklen_t clilen;
    char buffer[BUFFER_SIZE];
    int success = 0;

    /* Création de la socket TCP (SOCK_STREAM) utilisant le protocole IPv4 */
    sockfd = socket(AF_INET, SOCK_STREAM, 0);
    if (sockfd < 0) return -1;

    /* Option pour réutiliser l'adresse et le port immédiatement après 
       la fermeture du programme (évite l'erreur "Address already in use").
    */
    int opt = 1;
    setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));

    memset(&serv_addr, 0, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = htons(PORT);

    if (is_server) {
        /* * LOGIQUE SERVEUR
         * Attache la socket à toutes les interfaces locales et attend une connexion.
         */
        serv_addr.sin_addr.s_addr = INADDR_ANY;
        if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(serv_addr)) < 0) goto cleanup;
        
        listen(sockfd, 5);
        clilen = sizeof(cli_addr);
        newsockfd = accept(sockfd, (struct sockaddr *)&cli_addr, &clilen);
        if (newsockfd < 0) goto cleanup;

        /* Réception des données du client et extraction des valeurs 
           via sscanf pour les comparer aux données locales.
        */
        memset(buffer, 0, BUFFER_SIZE);
        if (recv(newsockfd, buffer, BUFFER_SIZE - 1, 0) > 0) {
            double c_tr, c_tr48, c_sdr, c_sdr48, c_debr, c_debr48;
            size_t c_maxr, c_minr;

            if (sscanf(buffer, "%lf,%lf,%lf,%lf,%lf,%lf,%zu,%zu", 
                &c_tr, &c_tr48, &c_sdr, &c_sdr48, &c_debr, &c_debr48, &c_maxr, &c_minr) == 8) {
                
                /* Affichage du tableau comparatif final SERVEUR vs CLIENT */
                printf("\n================================================================================\n");
                printf("                PERFORMANCE : SERVEUR & CLIENT\n");
                printf("================================================================================\n");
                printf("%-20s | %-26s | %-26s\n", "CARACTERISTIQUE", "SERVEUR (Local)", "CLIENT (Distant)");
                printf("---------------------|----------------------------|-----------------------------\n");
                
                /* Affichage des résultats pour RAND_R */
                printf("%-20s | %10.2f Mpts/s      | %10.2f Mpts/s\n", "VITESSE (Rand)", deb_r, c_debr);
                printf("%-20s | %10.4f            | %10.4f\n", "ECART-TYPE (Rand)", sd_r, c_sdr);
                printf("%-20s | %10zu fois        | %10zu fois\n", "OCCUR. MAX (Rand)", max_r, c_maxr);
                printf("%-20s | %10zu fois        | %10zu fois\n", "OCCUR. MIN (Rand)", min_r, c_minr);
                printf("%-20s | %10.4f s           | %10.4f s\n", "Temps Total (Rand)", t_r, c_tr);
                
                printf("---------------------|----------------------------|-----------------------------\n");
                
                /* Affichage des résultats pour DRAND48_R */
                printf("%-20s | %10.2f Mpts/s      | %10.2f Mpts/s\n", "VITESSE (R48)", deb_r48, c_debr48);
                printf("%-20s | %10.4f            | %10.4f\n", "ECART-TYPE (R48)", sd_r48, c_sdr48);
                printf("%-20s | %10zu fois        | %10zu fois\n", "OCCUR. MAX (R48)", max_r, c_maxr);
                printf("%-20s | %10zu fois        | %10zu fois\n", "OCCUR. MIN (R48)", min_r, c_minr);
                printf("%-20s | %10.4f s           | %10.4f s\n", "Temps Total (R48)", t_r48, c_tr48);

                printf("================================================================================\n");
                printf(" Analyse statistique et reseau terminee.\n");
                success = 1;
            }
        }
    } else {
        /* * LOGIQUE CLIENT
         * Se connecte au serveur local (localhost) et transmet ses propres statistiques.
         */
        inet_pton(AF_INET, "127.0.0.1", &serv_addr.sin_addr);
        if (connect(sockfd, (struct sockaddr *)&serv_addr, sizeof(serv_addr)) < 0) goto cleanup;

        /* Formatage des données en une chaîne CSV pour faciliter l'envoi */
        snprintf(buffer, BUFFER_SIZE, "%.6f,%.6f,%.6f,%.6f,%.6f,%.6f,%zu,%zu", 
                 t_r, t_r48, sd_r, sd_r48, deb_r, deb_r48, max_r, min_r);
        send(sockfd, buffer, strlen(buffer), 0);
        success = 1;
    }

cleanup:
    /* Fermeture propre des descripteurs de fichiers pour libérer les ressources système */
    if (newsockfd != -1) close(newsockfd);
    if (sockfd != -1) close(sockfd);
    return success ? 0 : -1;
}