#ifndef GENERATOR_H
#define GENERATOR_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <string.h>
#include <sys/statvfs.h>
#include <math.h>

/* * Définition des constantes globales du programme :
 * - NB_PROCESSUS : Nombre de coeurs/processus utilisés pour la génération.
 * - NB_ALEATOIRES_TOTAL : Volume total de données à générer (2 millions).
 * - SHM_NAME : Nom du segment de mémoire partagée POSIX.
 * - SIZE : Taille totale en octets de la mémoire à allouer.
 */
#define NB_PROCESSUS 4
#define NB_ALEATOIRES_TOTAL 2000000LL 
#define NB_ALEATOIRES_PAR_PROC (NB_ALEATOIRES_TOTAL / NB_PROCESSUS)
#define SHM_NAME "/mon_shm"
#define SIZE ((size_t)NB_ALEATOIRES_TOTAL * sizeof(int))
#define PORT 8080

/* * Structure Stats :
 * Regroupe les indicateurs de performance et les résultats de l'analyse
 * statistique des nombres générés (temps, écart-type, débit, etc.).
 */
typedef struct {
    double time;        
    double std_dev;     
    double throughput;  /* Débit (nombres générés par seconde) */
    size_t max_occ;     
    size_t min_occ;     
    int val_max;        
    int val_min;        
} Stats;

/* * generate_numbers :
 * Remplit une portion spécifique de la mémoire partagée avec des nombres aléatoires.
 * - shared_mem  : Pointeur vers le segment de mémoire partagée.
 * - start_index : Position de départ dans le tableau pour ce processus.
 * - count       : Nombre d'entiers à générer par ce processus.
 * - use_rand48  : Booléen (0 ou 1). Si 1, utilise drand48(), sinon utilise rand().
 */
void generate_numbers(int *shared_mem, size_t start_index, size_t count, int use_rand48);

/* * write_to_csv :
 * Enregistre les données contenues dans la mémoire partagée dans un fichier .csv.
 * - filename : Nom du fichier de destination.
 * - data     : Pointeur vers le tableau d'entiers à sauvegarder.
 * - count    : Nombre total d'éléments à écrire.
 * Retourne 0 en cas de succès, -1 en cas d'erreur d'ouverture de fichier.
 */
int write_to_csv(const char *filename, int *data, size_t count);

/* * get_elapsed_time :
 * Calcule la durée précise entre deux instants capturés par gettimeofday.
 * - start : Structure timeval contenant le temps de début.
 * - end   : Structure timeval contenant le temps de fin.
 * Retourne le temps écoulé en secondes (format double pour la précision).
 */
double get_elapsed_time(struct timeval start, struct timeval end);

/* * analyze_data :
 * Parcourt le tableau de données pour calculer les statistiques de performance.
 * - data       : Le tableau d'entiers générés à analyser.
 * - count      : La taille totale du tableau.
 * - algo_name  : Nom de l'algorithme utilisé (pour l'affichage).
 * - time_spent : Temps mis pour la génération (utilisé pour calculer le débit).
 * Retourne une structure Stats remplie avec les résultats.
 */
Stats analyze_data(int *data, size_t count, const char *algo_name, double time_spent);

/* * communicate_with_socket :
 * Gère l'échange de statistiques entre deux machines via le protocole TCP.
 * - is_server : Si 1, le programme attend une connexion ; si 0, il se connecte à l'IP.
 * - t_r, t_r48, etc. : Les différentes mesures (temps, écart-type, débit) 
 * pour les deux méthodes de génération (rand et rand48).
 * Retourne 0 en cas de succès, -1 en cas d'échec de communication.
 */
int communicate_with_socket(int is_server, double t_r, double t_r48, double sd_r, double sd_r48, double deb_r, double deb_r48, size_t max_r, size_t min_r);
#endif