#include "generator.h"
#include <sys/types.h>
#include <math.h>

/**
 * GÉNÉRATION DES NOMBRES (Version Réentrante)
 * Initialise les graines de génération de manière unique par processus
 * et remplit le segment de mémoire partagée selon l'algorithme choisi.
 */
void generate_numbers(int *shared_mem, size_t start_index, size_t count, int use_rand48) {
    /* Préparation des graines : utilisation du PID et de l'heure 
       pour garantir des séquences différentes entre les processus fils.
    */
    unsigned int seed_rand = (unsigned int)(getpid() ^ time(NULL));
    struct drand48_data buffer_rand48;
    
    srand48_r((long int)(getpid() ^ time(NULL)), &buffer_rand48);

    size_t i;
    for (i = 0; i < count; i++) {
        int nombre;
        
        /* Sélection de l'algorithme :
           - rand48_r : plus précis, utilise un état interne (buffer).
           - rand_r : standard, thread-safe.
        */
        if (use_rand48) {
            double res;
            drand48_r(&buffer_rand48, &res);
            nombre = (int)(res * 100);
        } else {
            nombre = rand_r(&seed_rand) % 100;
        }

        /* Vérification des limites pour éviter les débordements 
           de mémoire lors de l'écriture dans le segment partagé.
        */
        if (start_index + i < NB_ALEATOIRES_TOTAL) {
            shared_mem[start_index + i] = nombre;
        }
    }
    printf("Processus de PID %d terminé : %zu nombres générés.\n", getpid(), count);
}

/**
 * ÉCRITURE CSV OPTIMISÉE
 * Ouvre un fichier en écriture et utilise un buffer manuel 
 * pour limiter les accès disque lors de l'écriture de gros volumes.
 */
int write_to_csv(const char *filename, int *data, size_t count) {
    FILE *fp = fopen(filename, "w");
    if (fp == NULL) {
        perror("Erreur d'ouverture du fichier");
        return -1;
    }

    /* Optimisation : Allocation d'un tampon de 1 Mo pour réduire 
       les appels système d'écriture (write) vers le disque.
    */
    setvbuf(fp, NULL, _IOFBF, 1024 * 1024);

    fprintf(fp, "Index,Nombre\n");

    for (size_t i = 0; i < count; i++) {
        fprintf(fp, "%zu,%d\n", i, data[i]);
    }

    fclose(fp);
    return 0;
}

/**
 * CALCUL DE LA DURÉE ÉCOULÉE
 * Transforme la structure timeval en une valeur double en secondes.
 */
double get_elapsed_time(struct timeval start, struct timeval end) {
    return (double)(end.tv_sec - start.tv_sec) + 
           (double)(end.tv_usec - start.tv_usec) / 1000000.0;
}

/**
 * ANALYSE STATISTIQUE
 * Parcourt les données pour calculer la moyenne, l'écart-type, 
 * le débit ainsi que la distribution des occurrences.
 */
Stats analyze_data(int *data, size_t count, const char *algo_name, double time_spent) {
    size_t freq[100] = {0};
    double sum = 0;
    double sum_sq = 0;
    size_t pairs = 0;
    Stats s;

    /* Premier passage : accumulation des sommes pour les moments 
       statistiques et comptage des fréquences de chaque nombre.
    */
    for (size_t i = 0; i < count; i++) {
        int val = data[i];
        freq[val]++;
        sum += (double)val;
        sum_sq += (double)val * val;
        if (val % 2 == 0) pairs++;
    }

    double mean = sum / count;

    /* Calculs finaux : 
       - Écart-type via la formule de Koenig-Huygens.
       - Débit exprimé en millions de points par seconde (Mpts/s).
    */
    s.std_dev = sqrt((sum_sq / count) - (mean * mean));
    s.throughput = (count / 1000000.0) / time_spent;
    s.time = time_spent;

    s.max_occ = freq[0];
    s.min_occ = freq[0];
    s.val_max = 0;
    s.val_min = 0;

    /* Recherche des valeurs ayant les fréquences extrêmes 
       pour vérifier l'uniformité du générateur.
    */
    for (int i = 1; i < 100; i++) {
        if (freq[i] > s.max_occ) {
            s.max_occ = freq[i];
            s.val_max = i;
        }
        if (freq[i] < s.min_occ) {
            s.min_occ = freq[i];
            s.val_min = i;
        }
    }

    printf("\n --- ANALYSE : %s ---\n", algo_name);
    printf("Vitesse      : **%.2f Mpts/s** (Temps: %.4fs)\n", s.throughput, s.time);
    printf("Moyenne      : **%.4f** (Cible: 49.5000)\n", mean);
    printf("Écart-type   : **%.4f** (Cible: 28.8660)\n", s.std_dev);
    printf("Parité       : **%.2f%% Pairs**\n", (double)pairs/count*100.0);
    printf("Max          :**%d**--->%zu fois\n", s.val_max, s.max_occ);
    printf("Min          :**%d**---> %zu fois\n", s.val_min, s.min_occ);
    
    return s; 
}