import React, { useMemo, useState } from "react";
import mock from "../mockdata";
import TimeSeriesChart from "../components/TimeSeriesChart";
import ScatterPlot from "../components/ScatterPlot";
import PieChartPanel from "../components/PieChartPanel";
import KpiStrip from "../components/KpiStrip";
import {
	Select,
	SelectTrigger,
	SelectValue,
	SelectContent,
	SelectItem,
} from "../components/ui/select";
import { Input } from "../components/ui/input";

function getRegionName(id: number) {
	const r = mock.regions.find((x: any) => x.id === id);
	return r ? r.nom : `Region ${id}`;
}

export default function DashboardPage() {
	const [typeId, setTypeId] = useState<number>(2);
	const [startYear, setStartYear] = useState<number>(2018);
	const [endYear, setEndYear] = useState<number>(2022);
	const [deptId, setDeptId] = useState<number | null>(1);
	const [scatterYear, setScatterYear] = useState<number>(2022);

	// Time series: for each year, for each region compute average taux of communes in region for selected type
	const timeSeriesData = useMemo(() => {
		const years: number[] = [];
		for (let y = startYear; y <= endYear; y++) years.push(y);

		const regions = mock.regions.map((r: any) => r.nom);

		const rows = years.map((y) => {
			const row: any = { year: y };
			mock.regions.forEach((reg: any) => {
				const communesInRegion = mock.communes.filter((c: any) => {
					const dep = mock.departements.find(
						(d: any) => d.id === c.departementId,
					);
					return dep && dep.regionId === reg.id;
				});
				const taxes = mock.taxes.filter(
					(t: any) =>
						t.typeId === typeId &&
						t.annee === y &&
						communesInRegion.some((c: any) => c.id === t.communeId),
				);
				const avg = taxes.length
					? taxes.reduce((s: number, t: any) => s + t.taux, 0) /
						taxes.length
					: null;
				row[reg.nom] = avg;
			});
			return row;
		});

		return { rows, regions };
	}, [startYear, endYear, typeId]);

	// Scatter: communes in a departement for a given year
	const scatterData = useMemo(() => {
		if (!deptId) return [];
		const communesInDept = mock.communes.filter(
			(c: any) => c.departementId === deptId,
		);
		const taxes = mock.taxes.filter(
			(t: any) =>
				t.annee === scatterYear &&
				communesInDept.some((c: any) => c.id === t.communeId) &&
				t.typeId === typeId,
		);
		return taxes.map((t: any) => {
			const commune = mock.communes.find(
				(c: any) => c.id === t.communeId,
			);
			return {
				commune: commune?.nom || String(t.communeId),
				taux: t.taux,
				volume: t.volume,
			};
		});
	}, [deptId, scatterYear, typeId]);

	// Pie: sum volumes per region for given year and type
	const pieData = useMemo(() => {
		const year = scatterYear;
		const regions = mock.regions.map((r: any) => {
			const communesInRegion = mock.communes.filter((c: any) => {
				const dep = mock.departements.find(
					(d: any) => d.id === c.departementId,
				);
				return dep && dep.regionId === r.id;
			});
			const vols = mock.taxes.filter(
				(t: any) =>
					t.annee === year &&
					t.typeId === typeId &&
					communesInRegion.some((c: any) => c.id === t.communeId),
			);
			const sum = vols.reduce((s: number, t: any) => s + t.volume, 0);
			return { name: r.nom, value: sum };
		});
		return regions;
	}, [scatterYear, typeId]);

	// KPIs: total volume for selected endYear and delta vs previous year
	const kpis = useMemo(() => {
		const year = endYear;
		const prev = year - 1;
		const vols = mock.taxes
			.filter((t: any) => t.annee === year && t.typeId === typeId)
			.reduce((s: number, t: any) => s + t.volume, 0);
		const volsPrev = mock.taxes
			.filter((t: any) => t.annee === prev && t.typeId === typeId)
			.reduce((s: number, t: any) => s + t.volume, 0);
		const delta = volsPrev ? (vols - volsPrev) / volsPrev : 0;
		const avgTaux = (() => {
			const taxes = mock.taxes.filter(
				(t: any) => t.annee === year && t.typeId === typeId,
			);
			return taxes.length
				? taxes.reduce((s: number, t: any) => s + t.taux, 0) /
						taxes.length
				: 0;
		})();

		return [
			{
				label: "Collected volume (year " + year + ")",
				value: Math.round(vols),
				delta,
			},
			{
				label: "Avg tax rate (year " + year + ")",
				value: Number((avgTaux * 100).toFixed(2)),
				delta: 0,
			},
			{ label: "Regions", value: mock.regions.length },
		];
	}, [endYear, typeId]);

	return (
		<div className="p-4">
			<h2 className="text-2xl font-semibold mb-4">
				Fiscal Policy Dashboard (mock)
			</h2>

			<section className="mb-6 flex flex-wrap items-center gap-3">
				<label className="text-sm">Tax Type:</label>
				<Select value={String(typeId)} onValueChange={(v) => setTypeId(Number(v))}>
					<SelectTrigger className="w-64">
						<SelectValue placeholder="Select tax" />
					</SelectTrigger>
					<SelectContent>
						{mock.typeTaxes.map((tt: any) => (
							<SelectItem key={tt.id} value={String(tt.id)}>
								{tt.code} — {tt.label}
							</SelectItem>
						))}
					</SelectContent>
				</Select>

				<label className="text-sm ml-2">Years:</label>
				<Input className="w-20" type="number" value={String(startYear)} onChange={(e) => setStartYear(Number(e.target.value))} />
				<Input className="w-20 ml-2" type="number" value={String(endYear)} onChange={(e) => setEndYear(Number(e.target.value))} />
			</section>

			<section className="mb-6 bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<KpiStrip metrics={kpis} />

				<h3 className="text-lg font-medium mb-2">
					Time Series — average taux per region
				</h3>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<TimeSeriesChart
						data={timeSeriesData.rows}
						regions={timeSeriesData.regions}
					/>
				</div>
			</section>

			<section className="mb-6 bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Scatter — communes (department)
				</h3>
				<div className="flex items-center gap-2 mb-3">
					<label className="text-sm">Department:</label>
					<Select value={deptId ? String(deptId) : ''} onValueChange={(v) => setDeptId(v ? Number(v) : null)}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Select dept" />
						</SelectTrigger>
						<SelectContent>
							{mock.departements.map((d: any) => (
								<SelectItem key={d.id} value={String(d.id)}>
									{d.code} — {d.nom}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Year:</label>
					<Input className="w-28" type="number" value={String(scatterYear)} onChange={(e) => setScatterYear(Number(e.target.value))} />
				</div>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<ScatterPlot data={scatterData} />
				</div>
			</section>

			<section className="bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Pie — collected volume by region (year)
				</h3>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<PieChartPanel data={pieData} />
				</div>
			</section>
		</div>
	);
}
