import  { useMemo, useState, useEffect } from "react";
import TimeSeriesChart from "../components/TimeSeriesChart";
import ScatterPlot from "../components/ScatterPlot";
import PieChartPanel from "../components/PieChartPanel";
import { Loader2 } from "lucide-react";
import {
	Select,
	SelectTrigger,
	SelectValue,
	SelectContent,
	SelectItem,
} from "../components/ui/select";

export default function DashboardPage() {
	// TimeSeries filters (top controls)
	const [tsTypeId, setTsTypeId] = useState<number>(2);
	const [tsStartYear, setTsStartYear] = useState<number>(2018);
	const [tsEndYear, setTsEndYear] = useState<number>(2022);

	// Scatter filters (per-section controls)
	const [scTypeId, setScTypeId] = useState<number>(2);
	const [scDeptId, setScDeptId] = useState<number | null>(1);
	const [scYear, setScYear] = useState<number>(2022);

	// Pie chart filters (per-section controls)
	const [pieTypeId, setPieTypeId] = useState<number>(2);
	const [pieYear, setPieYear] = useState<number>(2022);

	// State for API data for Time Series
	const [rawTimeSeries, setRawTimeSeries] = useState<any[]>([]);
	const [isLoadingTimeSeries, setIsLoadingTimeSeries] = useState<boolean>(false);

	// State for API data for Scatter Plot
	const [rawScatterData, setRawScatterData] = useState<any[]>([]);
	const [isLoadingScatter, setIsLoadingScatter] = useState<boolean>(false);

	// State for API data for Pie Chart
	const [rawPieData, setRawPieData] = useState<any[]>([]);
	const [isLoadingPie, setIsLoadingPie] = useState<boolean>(false);

	// State for Lists (Departments, Tax Types)
	const [listTaxTypes, setListTaxTypes] = useState<any[]>([]);
	const [listDepartements, setListDepartements] = useState<any[]>([]);

	// Fetch Lists
	useEffect(() => {
		// Fetch Tax Types
		fetch("https://localhost/type_taxes", {
			headers: { Accept: "application/ld+json" },
		})
			.then((res) => res.json())
			.then((data) => {
				if (data["member"]) {
					setListTaxTypes(data["member"]);
					// Default to first type if available and no selection
					// or ensure defaults are valid
				}
			})
			.catch((err) => console.error("Failed to fetch tax types:", err));

		// Fetch Departements
		fetch("https://localhost/departements?pagination=false", {
			headers: { Accept: "application/ld+json" },
		})
			.then((res) => res.json())
			.then((data) => {
				if (data["member"]) {
					setListDepartements(
						data["member"].sort((a: any, b: any) =>
							a.code - b.code
						)
					);
				}
			})
			.catch((err) => console.error("Failed to fetch departements:", err));
	}, []);

	// Fetch Time Series data from API
	useEffect(() => {
		if (listTaxTypes.length === 0) return;
		const typeObj = listTaxTypes.find((t: any) => t.id === tsTypeId);
		if (!typeObj) return;

		const query = new URLSearchParams({
			startYear: String(tsStartYear),
			endYear: String(tsEndYear),
			typeCode: typeObj.code,
		});

		const fetchData = async () => {
			setIsLoadingTimeSeries(true);
			try {
				const res = await fetch(`https://localhost/stats/average-tax-by-region?${query.toString()}`, {
					headers: {
						Accept: "application/ld+json",
					},
				});
				const data = await res.json();
				if (data["member"]) {
					setRawTimeSeries(data["member"]);
				} else {
					setRawTimeSeries([]);
				}
			} catch (err) {
				console.error("Failed to fetch time series:", err);
				setRawTimeSeries([]);
			} finally {
				setIsLoadingTimeSeries(false);
			}
		};

		fetchData();
	}, [tsStartYear, tsEndYear, tsTypeId, listTaxTypes]);

	// Fetch Scatter data from API
	useEffect(() => {
		if (!scDeptId || listTaxTypes.length === 0) return;
		const typeObj = listTaxTypes.find((t: any) => t.id === scTypeId);
		if (!typeObj) return;

		const query = new URLSearchParams({
			annee: String(scYear),
			"commune.departement": `/api/departements/${scDeptId}`,
			"type.code": typeObj.code,
			pagination: "false",
		});

		const fetchData = async () => {
			setIsLoadingScatter(true);
			try {
				const res = await fetch(`https://localhost/taxes?${query.toString()}`, {
					headers: {
						Accept: "application/ld+json",
					},
				});
				const data = await res.json();
				if (data["member"]) {
					setRawScatterData(data["member"]);
				} else {
					setRawScatterData([]);
				}
			} catch (err) {
				console.error("Failed to fetch scatter data:", err);
				setRawScatterData([]);
			} finally {
				setIsLoadingScatter(false);
			}
		};

		fetchData();
	}, [scYear, scDeptId, scTypeId]);

	// Fetch Pie data from API
	useEffect(() => {
		if (listTaxTypes.length === 0) return;
		const typeObj = listTaxTypes.find((t: any) => t.id === pieTypeId);
		if (!typeObj) return;

		const query = new URLSearchParams({
			year: String(pieYear),
			typeCode: typeObj.code,
		});

		const fetchData = async () => {
			setIsLoadingPie(true);
			try {
				const res = await fetch(`https://localhost/stats/total-volume-by-region?${query.toString()}`, {
					headers: {
						Accept: "application/ld+json",
					},
				});
				const data = await res.json();
				if (data["member"]) {
					setRawPieData(data["member"]);
				} else {
					setRawPieData([]);
				}
			} catch (err) {
				console.error("Failed to fetch pie data:", err);
				setRawPieData([]);
			} finally {
				setIsLoadingPie(false);
			}
		};

		fetchData();
	}, [pieYear, pieTypeId, listTaxTypes]);

	// Time series: transform API data for chart
	const timeSeriesData = useMemo(() => {
		if (!rawTimeSeries.length) return { rows: [], regions: [] };

		const regions = Array.from(
			new Set(rawTimeSeries.map((item: any) => item.region)),
		);
		const availableYears = Array.from(
			new Set(rawTimeSeries.map((item: any) => item.annee)),
		).sort((a: any, b: any) => a - b);

		const rows = availableYears.map((y: any) => {
			const row: any = { year: y };
			regions.forEach((reg: any) => {
				const entry = rawTimeSeries.find(
					(item: any) => item.annee === y && item.region === reg,
				);
				row[reg] = entry ? entry.tauxMoyen : null;
			});
			return row;
		});

		return { rows, regions };
	}, [rawTimeSeries]);

	// Scatter: transform API data
	const scatterData = useMemo(() => {
		return rawScatterData.map((t: any) => {
			let name = "Commune";
			if (t.commune && typeof t.commune === "object" && t.commune.nom) {
				name = t.commune.nom;
			} else if (typeof t.commune === "string") {
				// Try to extract ID from IRI and match with mock if real name not available
				const parts = t.commune.split("/");
				const id = parts[parts.length - 1];
				name = `Commune ${id}`;
			}

			return {
				commune: name,
				taux: t.taux,
				volume: t.volume,
			};
		});
	}, [rawScatterData]);

	// Pie: transform API data
	const pieData = useMemo(() => {
		return rawPieData.map((d: any) => ({
			name: d.region,
			value: d.totalVolume,
		}));
	}, [rawPieData]);

	// available years for selectors (hardcoded per request)
	const years = [2018, 2019, 2020, 2021, 2022];

	return (
		<div className="p-4 space-y-4">
			<h2 className="text-2xl font-bold tracking-tight">Tableau de bord fiscal</h2>

			<section className="bg-slate-50 dark:bg-slate-900 p-4 rounded-lg border">
				<h3 className="text-lg font-medium mb-4">
					Séries temporelles - taux moyen par région
				</h3>

				<div className="flex flex-wrap items-center gap-4 mb-4">
					<div className="flex items-center gap-2">
						<label className="text-sm font-medium">Type de taxe :</label>
						<Select value={String(tsTypeId)} onValueChange={(v) => setTsTypeId(Number(v))}>
							<SelectTrigger className="w-[280px]">
								<SelectValue placeholder="Sélectionner un type" />
							</SelectTrigger>
							<SelectContent>
								{listTaxTypes.map((tt: any) => (
									<SelectItem key={tt.id} value={String(tt.id)}>
										{tt.code} - {tt.label}
									</SelectItem>
								))}
							</SelectContent>
						</Select>
					</div>

					<div className="flex items-center gap-2">
						<label className="text-sm font-medium">Période :</label>
						<Select value={String(tsStartYear)} onValueChange={(v) => {
							const y = Number(v);
							if (y > tsEndYear) {
								setTsStartYear(y);
								setTsEndYear(y);
							} else {
								setTsStartYear(y);
							}
						}}>
							<SelectTrigger className="w-[100px]">
								<SelectValue placeholder="Début" />
							</SelectTrigger>
							<SelectContent>
								{years.map((y: number) => (
									<SelectItem key={y} value={String(y)}>{y}</SelectItem>
								))}
							</SelectContent>
						</Select>
						<span className="text-muted-foreground">-</span>
						<Select value={String(tsEndYear)} onValueChange={(v) => {
							const y = Number(v);
							if (y < tsStartYear) {
								setTsStartYear(y);
								setTsEndYear(y);
							} else {
								setTsEndYear(y);
							}
						}}>
							<SelectTrigger className="w-[100px]">
								<SelectValue placeholder="Fin" />
							</SelectTrigger>
							<SelectContent>
								{years.map((y: number) => (
									<SelectItem key={y} value={String(y)}>{y}</SelectItem>
								))}
							</SelectContent>
						</Select>
					</div>
				</div>

				<div className="bg-white dark:bg-slate-800 p-3 rounded min-h-[300px]">
					{isLoadingTimeSeries ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<Loader2 className="h-8 w-8 animate-spin text-blue-500 mb-2" />
							<span className="text-sm text-gray-500">Chargement des données...</span>
						</div>
					) : rawTimeSeries.length === 0 ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<span className="text-sm text-gray-500">Aucune donnée disponible pour cette sélection.</span>
						</div>
					) : (
						<TimeSeriesChart
							data={timeSeriesData.rows}
							regions={timeSeriesData.regions}
						/>
					)}
				</div>
			</section>

			<section className="mb-6 bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Nuage de points - communes (département)
				</h3>
				<div className="flex items-center gap-2 mb-3">
					<label className="text-sm">Type de taxe :</label>
					<Select value={String(scTypeId)} onValueChange={(v) => setScTypeId(Number(v))}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un type" />
						</SelectTrigger>
						<SelectContent>
							{listTaxTypes.map((tt: any) => (
								<SelectItem key={tt.id} value={String(tt.id)}>
									{tt.code} - {tt.label}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Département :</label>
					<Select value={scDeptId ? String(scDeptId) : ''} onValueChange={(v) => setScDeptId(v ? Number(v) : null)}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un département" />
						</SelectTrigger>
						<SelectContent>
							{listDepartements.map((d: any) => (
								<SelectItem key={d.id} value={String(d.id)}>
									{d.code} - {d.nom}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Année :</label>
					<Select value={String(scYear)} onValueChange={(v) => setScYear(Number(v))}>
						<SelectTrigger className="w-28">
							<SelectValue placeholder="Année" />
						</SelectTrigger>
						<SelectContent>
							{years.map((y: number) => (
								<SelectItem key={y} value={String(y)}>{y}</SelectItem>
							))}
						</SelectContent>
					</Select>
				</div>
				<div className="bg-white dark:bg-slate-800 p-3 rounded min-h-[300px]">
					{isLoadingScatter ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<Loader2 className="h-8 w-8 animate-spin text-blue-500 mb-2" />
							<span className="text-sm text-gray-500">Chargement des données...</span>
						</div>
					) : rawScatterData.length === 0 ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<span className="text-sm text-gray-500">
								Aucune donnée disponible pour cette sélection.
							</span>
						</div>
					) : (
						<ScatterPlot data={scatterData} />
					)}
				</div>
			</section>

			<section className="bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Diagramme en secteurs - volume collecté par région (année)
				</h3>
				<div className="mb-3 flex items-center gap-2">
					<label className="text-sm">Type de taxe :</label>
					<Select value={String(pieTypeId)} onValueChange={(v) => setPieTypeId(Number(v))}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un type" />
						</SelectTrigger>
						<SelectContent>
							{listTaxTypes.map((tt: any) => (
								<SelectItem key={tt.id} value={String(tt.id)}>
									{tt.code} - {tt.label}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Année :</label>
					<Select value={String(pieYear)} onValueChange={(v) => setPieYear(Number(v))}>
						<SelectTrigger className="w-28">
							<SelectValue placeholder="Année" />
						</SelectTrigger>
						<SelectContent>
							{years.map((y: number) => (
								<SelectItem key={y} value={String(y)}>{y}</SelectItem>
							))}
						</SelectContent>
					</Select>
				</div>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					{isLoadingPie ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<Loader2 className="h-8 w-8 animate-spin text-blue-500 mb-2" />
							<span className="text-sm text-gray-500">Chargement des données...</span>
						</div>
					) : rawPieData.length === 0 ? (
						<div className="flex flex-col items-center justify-center h-full min-h-[300px]">
							<span className="text-sm text-gray-500">
								Aucune donnée disponible pour cette sélection.
							</span>
						</div>
					) : (
						<PieChartPanel data={pieData} />
					)}
				</div>
			</section>
		</div>
	);
}
