import  { useMemo, useState } from "react";
import mock from "../mockdata";
import TimeSeriesChart from "../components/TimeSeriesChart";
import ScatterPlot from "../components/ScatterPlot";
import PieChartPanel from "../components/PieChartPanel";
import KpiStrip from "../components/KpiStrip";
import {
	Select,
	SelectTrigger,
	SelectValue,
	SelectContent,
	SelectItem,
} from "../components/ui/select";
import { Input } from "../components/ui/input";



export default function DashboardPage() {
	// TimeSeries filters (top controls)
	const [tsTypeId, setTsTypeId] = useState<number>(2);
	const [tsStartYear, setTsStartYear] = useState<number>(2018);
	const [tsEndYear, setTsEndYear] = useState<number>(2022);

	// Scatter filters (per-section controls)
	const [scTypeId, setScTypeId] = useState<number>(2);
	const [scDeptId, setScDeptId] = useState<number | null>(1);
	const [scYear, setScYear] = useState<number>(2022);

	// Pie chart filters (per-section controls)
	const [pieTypeId, setPieTypeId] = useState<number>(2);
	const [pieYear, setPieYear] = useState<number>(2022);

	// Time series: for each year, for each region compute average taux of communes in region for selected type
	const timeSeriesData = useMemo(() => {
		const years: number[] = [];
		for (let y = tsStartYear; y <= tsEndYear; y++) years.push(y);

		const regions = mock.regions.map((r: any) => r.nom);

		const rows = years.map((y) => {
			const row: any = { year: y };
			mock.regions.forEach((reg: any) => {
				const communesInRegion = mock.communes.filter((c: any) => {
					const dep = mock.departements.find(
						(d: any) => d.id === c.departementId,
					);
					return dep && dep.regionId === reg.id;
				});
				const taxes = mock.taxes.filter(
					(t: any) =>
						t.typeId === tsTypeId &&
						t.annee === y &&
						communesInRegion.some((c: any) => c.id === t.communeId),
				);
				const avg = taxes.length
					? taxes.reduce((s: number, t: any) => s + t.taux, 0) /
						taxes.length
					: null;
				row[reg.nom] = avg;
			});
			return row;
		});

		return { rows, regions };
	}, [tsStartYear, tsEndYear, tsTypeId]);

	// Scatter: communes in a departement for a given year
	const scatterData = useMemo(() => {
		if (!scDeptId) return [];
		const communesInDept = mock.communes.filter(
			(c: any) => c.departementId === scDeptId,
		);
		const taxes = mock.taxes.filter(
			(t: any) =>
				t.annee === scYear &&
				communesInDept.some((c: any) => c.id === t.communeId) &&
				t.typeId === scTypeId,
		);
		return taxes.map((t: any) => {
			const commune = mock.communes.find(
				(c: any) => c.id === t.communeId,
			);
			return {
				commune: commune?.nom || String(t.communeId),
				taux: t.taux,
				volume: t.volume,
			};
		});
	}, [scDeptId, scYear, scTypeId]);

	// Pie: sum volumes per region for given year and type
	const pieData = useMemo(() => {
		const year = pieYear;
		const regions = mock.regions.map((r: any) => {
			const communesInRegion = mock.communes.filter((c: any) => {
				const dep = mock.departements.find(
					(d: any) => d.id === c.departementId,
				);
				return dep && dep.regionId === r.id;
			});
			const vols = mock.taxes.filter(
				(t: any) =>
					t.annee === year &&
					t.typeId === pieTypeId &&
					communesInRegion.some((c: any) => c.id === t.communeId),
			);
			const sum = vols.reduce((s: number, t: any) => s + t.volume, 0);
			return { name: r.nom, value: sum };
		});
		return regions;
	}, [pieYear, pieTypeId]);

	// KPIs: total volume for selected endYear and delta vs previous year
	const kpis = useMemo(() => {
		const year = tsEndYear;
		const prev = year - 1;
		const vols = mock.taxes
			.filter((t: any) => t.annee === year && t.typeId === tsTypeId)
			.reduce((s: number, t: any) => s + t.volume, 0);
		const volsPrev = mock.taxes
			.filter((t: any) => t.annee === prev && t.typeId === tsTypeId)
			.reduce((s: number, t: any) => s + t.volume, 0);
		const delta = volsPrev ? (vols - volsPrev) / volsPrev : 0;
		const avgTaux = (() => {
			const taxes = mock.taxes.filter(
				(t: any) => t.annee === year && t.typeId === tsTypeId,
			);
			return taxes.length
				? taxes.reduce((s: number, t: any) => s + t.taux, 0) /
						taxes.length
				: 0;
		})();

		return [
			{
				label: "Volume collecté (" + year + ")",
				value: Math.round(vols),
				delta,
			},
			{
				label: "Taux moyen (" + year + ")",
				value: Number((avgTaux * 100).toFixed(2)),
				delta: 0,
			},
			{ label: "Régions", value: mock.regions.length },
		];
	}, [tsEndYear, tsTypeId]);

	return (
		<div className="p-4">
			<h2 className="text-2xl font-semibold mb-4">
				Tableau de bord fiscal
			</h2>

			<section className="mb-6 flex flex-wrap items-center gap-3">
				<label className="text-sm">Type de taxe :</label>
				<Select value={String(tsTypeId)} onValueChange={(v) => setTsTypeId(Number(v))}>
					<SelectTrigger className="w-64">
						<SelectValue placeholder="Sélectionner un type" />
					</SelectTrigger>
					<SelectContent>
						{mock.typeTaxes.map((tt: any) => (
							<SelectItem key={tt.id} value={String(tt.id)}>
								{tt.code} — {tt.label}
							</SelectItem>
						))}
					</SelectContent>
				</Select>

				<label className="text-sm ml-2">Années :</label>
				<Input className="w-20" type="number" value={String(tsStartYear)} onChange={(e) => setTsStartYear(Number(e.target.value))} />
				<Input className="w-20 ml-2" type="number" value={String(tsEndYear)} onChange={(e) => setTsEndYear(Number(e.target.value))} />
			</section>

			<section className="mb-6 bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<KpiStrip metrics={kpis} />

				<h3 className="text-lg font-medium mb-2">
					Séries temporelles — taux moyen par région
				</h3>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<TimeSeriesChart
						data={timeSeriesData.rows}
						regions={timeSeriesData.regions}
					/>
				</div>
			</section>

			<section className="mb-6 bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Nuage de points — communes (département)
				</h3>
				<div className="flex items-center gap-2 mb-3">
					<label className="text-sm">Type de taxe :</label>
					<Select value={String(scTypeId)} onValueChange={(v) => setScTypeId(Number(v))}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un type" />
						</SelectTrigger>
						<SelectContent>
							{mock.typeTaxes.map((tt: any) => (
								<SelectItem key={tt.id} value={String(tt.id)}>
									{tt.code} — {tt.label}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Département :</label>
					<Select value={scDeptId ? String(scDeptId) : ''} onValueChange={(v) => setScDeptId(v ? Number(v) : null)}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un département" />
						</SelectTrigger>
						<SelectContent>
							{mock.departements.map((d: any) => (
								<SelectItem key={d.id} value={String(d.id)}>
									{d.code} — {d.nom}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Année :</label>
					<Input className="w-28" type="number" value={String(scYear)} onChange={(e) => setScYear(Number(e.target.value))} />
				</div>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<ScatterPlot data={scatterData} />
				</div>
			</section>

			<section className="bg-slate-50 dark:bg-slate-900 p-4 rounded-lg">
				<h3 className="text-lg font-medium mb-2">
					Diagramme en secteurs — volume collecté par région (année)
				</h3>
				<div className="mb-3 flex items-center gap-2">
					<label className="text-sm">Type de taxe :</label>
					<Select value={String(pieTypeId)} onValueChange={(v) => setPieTypeId(Number(v))}>
						<SelectTrigger className="w-56">
							<SelectValue placeholder="Sélectionner un type" />
						</SelectTrigger>
						<SelectContent>
							{mock.typeTaxes.map((tt: any) => (
								<SelectItem key={tt.id} value={String(tt.id)}>
									{tt.code} — {tt.label}
								</SelectItem>
							))}
						</SelectContent>
					</Select>
					<label className="text-sm ml-2">Année :</label>
					<Input className="w-28" type="number" value={String(pieYear)} onChange={(e) => setPieYear(Number(e.target.value))} />
				</div>
				<div className="bg-white dark:bg-slate-800 p-3 rounded">
					<PieChartPanel data={pieData} />
				</div>
			</section>
		</div>
	);
}
