<?php

namespace App\Repository;

use App\Entity\Taxe;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Taxe>
 */
class TaxeRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Taxe::class);
    }

    /**
     * Récupère les taux moyens par région pour une plage d'années et un type de taxe.
     *
     * @param int $startYear
     * @param int $endYear
     * @param string $typeCode
     * @return array
     */
    public function findAverageTauxByRegion(int $startYear, int $endYear, string $typeCode): array
    {
        return $this->createQueryBuilder('t')
            ->select('r.nom as region', 't.annee', 'AVG(t.taux) as tauxMoyen')
            ->join('t.commune', 'c')
            ->join('c.departement', 'd')
            ->join('d.region', 'r')
            ->join('t.type', 'typeTaxe')
            ->where('t.annee >= :start')
            ->andWhere('t.annee <= :end')
            ->andWhere('typeTaxe.code = :typeCode')
            ->groupBy('r.nom', 't.annee')
            ->orderBy('t.annee', 'ASC')
            ->addOrderBy('r.nom', 'ASC')
            ->setParameter('start', $startYear)
            ->setParameter('end', $endYear)
            ->setParameter('typeCode', $typeCode)
            ->getQuery()
            ->getResult();
    }

    /**
     * Récupère la somme des volumes par région pour une année et un type de taxe.
     *
     * @param int $annee
     * @param string $typeCode
     * @return array
     */
    public function findVolumesByRegion(int $annee, string $typeCode): array
    {
        return $this->createQueryBuilder('t')
            ->select('r.nom as region', 'SUM(t.volume) as totalVolume')
            ->join('t.commune', 'c')
            ->join('c.departement', 'd')
            ->join('d.region', 'r')
            ->join('t.type', 'typeTaxe')
            ->where('t.annee = :annee')
            ->andWhere('typeTaxe.code = :typeCode')
            ->groupBy('r.nom')
            ->orderBy('totalVolume', 'DESC')
            ->setParameter('annee', $annee)
            ->setParameter('typeCode', $typeCode)
            ->getQuery()
            ->getResult();
    }
}
