<?php

namespace App\DataFixtures;

use App\Entity\Commune;
use App\Entity\Departement;
use App\Entity\Region;
use App\Entity\Taxe;
use App\Entity\TypeTaxe;
use App\Service\CommuneService;
use App\Service\DepartementService;
use App\Service\RegionService;
use App\Service\TypeTaxeService;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;

class AppFixtures extends Fixture
{
	public function __construct(
		private RegionService $regionService,
		private DepartementService $departementService,
		private CommuneService $communeService,
		private TypeTaxeService $typeTaxeService,
	) {}

	const DATA_FILES = [
		'data/REI_2018.csv',
		'data/REI_COMPLET_2019.csv',
		'data/REI_COMPLET_2020.csv',
		'data/REI_2021.csv',
		'data/REI_2022.csv',
	];

	public function load(ObjectManager $manager): void {
		ini_set('memory_limit', -1); // les fichiers excels sont tellement énormes qu'il faut beaucoup de RAM

		$this->loadCsv($manager);

		$manager->flush();

		ini_set('memory_limit', '1024M');
	}

	public function loadCsv(ObjectManager $manager): void {
		// $pool = new ApcuCachePool();
		// $simpleCache = new SimpleCacheBridge($pool);

		// \PhpOffice\PhpSpreadsheet\Settings::setCache($simpleCache);

		$typeTFPNB = $this->typeTaxeService->findOneByCode(TypeTaxe::TFPNB_CODE);
		$typeTFPB = $this->typeTaxeService->findOneByCode(TypeTaxe::TFPB_CODE);
		$typeTH = $this->typeTaxeService->findOneByCode(TypeTaxe::TH_CODE);
		$typeCFE = $this->typeTaxeService->findOneByCode(TypeTaxe::CFE_CODE);

		if(is_null($typeTFPNB)) {
			$typeTFPNB = new TypeTaxe();
			$typeTFPNB->setCode(TypeTaxe::TFPNB_CODE);
			$typeTFPNB->setLabel("Taxe foncière sur les propriétés non bâties");

			$manager->persist($typeTFPNB);
		}
		if(is_null($typeTFPB)) {
			$typeTFPB = new TypeTaxe();
			$typeTFPB->setCode(TypeTaxe::TFPB_CODE);
			$typeTFPB->setLabel("Taxe foncière sur les propriétés bâties");

			$manager->persist($typeTFPB);
		}
		if(is_null($typeTH)) {
			$typeTH = new TypeTaxe();
			$typeTH->setCode(TypeTaxe::TH_CODE);
			$typeTH->setLabel("Taxe d'habitation");

			$manager->persist($typeTH);
		}
		if(is_null($typeCFE)) {
			$typeCFE = new TypeTaxe();
			$typeCFE->setCode(TypeTaxe::CFE_CODE);
			$typeCFE->setLabel("Cotisation foncière des entreprises");

			$manager->persist($typeCFE);
		}

		foreach (self::DATA_FILES as $path) {
			$explodedPath = explode('_', str_replace('.csv', '', $path));
			$annee = intval(end($explodedPath));

			$depNumCol = null;
			$depNomCol = null;
			$regNomCol = null;
			$comNumCol = null;
			$comNomCol = null;
			$tauxTFPNBCol = null;
			$volTFPNBCol = null;
			$tauxTFPBCol = null;
			$volTFPBCol = null;
			$tauxTHCol = null;
			$volTHCol = null;
			$tauxCFECol = null;
			$volCFECol = null;

			$colsFound = false;

			switch ($annee) {
				case 2018:
					$colsFound = true;

					$depNumCol = 1;
					$depNomCol = 935;
					$regNomCol = 936;
					$comNumCol = 3;
					$comNomCol = 10;
					$tauxTFPNBCol = 13;
					$volTFPNBCol = 14;
					$tauxTFPBCol = 68;
					$volTFPBCol = 69;
					$tauxTHCol = 167;
					$volTHCol = 168;
					$tauxCFECol = 327;
					$volCFECol = 328;
					break;
				case 2019:
					$colsFound = true;

					$depNumCol = 1;
					$depNomCol = 935;
					$regNomCol = 936;
					$comNumCol = 3;
					$comNomCol = 10;
					$tauxTFPNBCol = 13;
					$volTFPNBCol = 14;
					$tauxTFPBCol = 68;
					$volTFPBCol = 69;
					$tauxTHCol = 167;
					$volTHCol = 168;
					$tauxCFECol = 327;
					$volCFECol = 328;
					break;
				case 2020:
					$colsFound = true;

					$depNumCol = 1;
					$depNomCol = 1066;
					$regNomCol = 1067;
					$comNumCol = 3;
					$comNomCol = 10;
					$tauxTFPNBCol = 13;
					$volTFPNBCol = 14;
					$tauxTFPBCol = 68;
					$volTFPBCol = 69;
					$tauxTHCol = 185;
					$volTHCol = 186;
					$tauxCFECol = 347;
					$volCFECol = 348;
					break;
				case 2021:
					$colsFound = true;

					$depNumCol = 1;
					$depNomCol = 1108;
					$regNomCol = 1109;
					$comNumCol = 3;
					$comNomCol = 10;
					$tauxTFPNBCol = 13;
					$volTFPNBCol = 14;
					$tauxTFPBCol = 68;
					$volTFPBCol = 69;
					$tauxTHCol = 171;
					$volTHCol = 172;
					$tauxCFECol = 322;
					$volCFECol = 323;
					break;
				case 2022:
					$colsFound = true;

					$depNumCol = 1;
					$depNomCol = 1145;
					$regNomCol = 1146;
					$comNumCol = 3;
					$comNomCol = 10;
					$tauxTFPNBCol = 13;
					$volTFPNBCol = 14;
					$tauxTFPBCol = 68;
					$volTFPBCol = 69;
					$tauxTHCol = 175;
					$volTHCol = 176;
					$tauxCFECol = 332;
					$volCFECol = 333;
					break;
			}

			if (!$colsFound) {
				continue;
			}

			$titleLine = true;
			if (($handle = fopen($path, "r")) !== FALSE) {
				while (($data = fgetcsv($handle, separator: ",")) !== FALSE) {
					if ($titleLine) {
						$titleLine = false;
						continue;
					}

					$depNum = intval($data[$depNumCol - 1] ?? null);
					$depNom = $data[$depNomCol - 1] ?? null;
					$regNom = $data[$regNomCol - 1] ?? null;
					$comNum = intval($data[$comNumCol - 1] ?? null);
					$comNom = $data[$comNomCol - 1] ?? null;
					$tauxTFPNB = floatval($data[$tauxTFPNBCol - 1] ?? null);
					$volTFPNB = floatval($data[$volTFPNBCol - 1] ?? null);
					$tauxTFPB = floatval($data[$tauxTFPBCol - 1] ?? null);
					$volTFPB = floatval($data[$volTFPBCol - 1] ?? null);
					$tauxTH = floatval($data[$tauxTHCol - 1] ?? null);
					$volTH = floatval($data[$volTHCol - 1] ?? null);
					$tauxCFE = floatval($data[$tauxCFECol - 1] ?? null);
					$volCFE = floatval($data[$volCFECol - 1] ?? null);

					$args = [
						$depNum, $depNom, $regNom, $comNum, $comNom, $tauxTFPNB, $volTFPNB,
						$tauxTFPB, $volTFPB, $tauxTH, $volTH, $tauxCFE, $volCFE
					];

					if(in_array(null, $args)) { continue; }

					$departement = $this->departementService->findOneByCode($depNum);
					$commune = $this->communeService->findOneByCode($comNum);

					if(is_null($departement)) {
						$region = $this->regionService->findOneByNom($regNom);

						if(is_null($region)) {
							$region = new Region();
							$region->setNom($regNom);

							$manager->persist($region);
							$manager->flush();
							$region = $this->regionService->findOneByNom($regNom);
						}

						$departement = new Departement();
						$departement->setCode($depNum);
						$departement->setNom($depNom);
						$departement->setRegion($region);

						$manager->persist($departement);
						$manager->flush();
						$departement = $this->departementService->findOneByCode($depNum);
					}

					if (is_null($commune)) {
						$commune = new Commune();
						$commune->setCode($comNum);
						$commune->setNom($comNom);
						$commune->setDepartement($departement);

						$manager->persist($commune);
						$manager->flush();
						$commune = $this->communeService->findOneByCode($comNum);
					}

					$taxeTFPNB = new Taxe();
					$taxeTFPNB->setType($typeTFPNB);
					$taxeTFPNB->setCommune($commune);
					$taxeTFPNB->setTaux($tauxTFPNB);
					$taxeTFPNB->setVolume($volTFPNB);
                    $taxeTFPNB->setAnnee($annee);
                    $manager->persist($taxeTFPNB);

					$taxeTFPB = new Taxe();
					$taxeTFPB->setType($typeTFPB);
					$taxeTFPB->setCommune($commune);
					$taxeTFPB->setTaux($tauxTFPB);
					$taxeTFPB->setVolume($volTFPB);
                    $taxeTFPB->setAnnee($annee);
                    $manager->persist($taxeTFPB);

					$taxeTH = new Taxe();
					$taxeTH->setType($typeTH);
					$taxeTH->setCommune($commune);
					$taxeTH->setTaux($tauxTH);
					$taxeTH->setVolume($volTH);
                    $taxeTH->setAnnee($annee);
                    $manager->persist($taxeTH);

					$taxeCFE = new Taxe();
					$taxeCFE->setType($typeCFE);
					$taxeCFE->setCommune($commune);
					$taxeCFE->setTaux($tauxCFE);
					$taxeCFE->setVolume($volCFE);
                    $taxeCFE->setAnnee($annee);
                    $manager->persist($taxeCFE);
				}
				fclose($handle);
			}
		}
	}
}
