import { LineChart, Line, XAxis, YAxis, Legend, CartesianGrid } from "recharts";
import {
	ChartContainer,
	ChartTooltipContent,
	ChartTooltip,
} from "@/components/ui/chart";

type SeriesDatum = { year: number; [key: string]: number | string };

export default function TimeSeriesChart({
	data,
	regions,
	colors = [] as string[],
}: {
	data: SeriesDatum[]; // each item: { year, RegionA: value, RegionB: value }
	regions: string[]; // region keys to plot
	colors?: string[];
}) {
	const palette = colors.length
		? colors
		: [
				"#0B5FFF",
				"#00A78E",
				"#FFB020",
				"#6B21A8",
				"#475569",
				"#EF4444",
				"#10B981",
				"#6366F1",
				"#F59E0B",
				"#E11D48",
				"#06B6D4",
				"#8B5CF6",
			];

	const config = Object.fromEntries(
		regions.map((r, i) => [
			r,
			{ label: r, color: palette[i % palette.length] },
		]),
	);

	return (
		<ChartContainer config={config}>
			<LineChart
				data={data}
				margin={{ top: 12, right: 120, left: 0, bottom: 12 }}
			>
				<CartesianGrid strokeDasharray="3 3" />
				<XAxis
					dataKey="year"
					label={{
						value: "Année",
						position: "insideBottom",
						offset: -5,
					}}
				/>
				<YAxis />
				<ChartTooltip content={<ChartTooltipContent />} />
				<Legend verticalAlign="middle" align="right" layout="vertical" wrapperStyle={{ right: -20 }} />
				{regions.map((r, i) => (
					<Line
						key={r}
						type="monotone"
						dataKey={r}
						stroke={
							palette[i % palette.length]
						}
						strokeWidth={2}
						dot={{ r: 2 }}
						activeDot={{ r: 4 }}
						connectNulls
					/>
				))}
			</LineChart>
		</ChartContainer>
	);
}
