import React, { useState } from 'react';
import { gql, useQuery, useMutation } from "@apollo/client";
import { useHistory } from "react-router-dom";
import { GraphQLID } from 'graphql';

import ListAnnonceItem from '../components/list-annonce-item';
import { API_URL } from './../constants';

const GET_ANNONCES_QUERY = gql`
    query getAnnonces {
    annonces {
        _id
        title
        type
        status
        published
        price
        description
        availabilityDate
        coverImage
        images
        assessments {
        _id
        text  
        annonceId
        }
    }
    }
`;

const DELETE_ANNONCE_MUTATION = gql`
  mutation deleteAnnonce($annonceId: ID!) {
    deleteAnnonce(annonceId: $annonceId) {
      _id
      title
      type
      status
      published
      price
      description
      availabilityDate
      assessments {
        _id
        text
      }
    }
  }
`;

const ListAnnonce = () => {
    let history = useHistory();

    const { loading, error, data, updateQuery } = useQuery(GET_ANNONCES_QUERY);

    const [showModal, setShowModal] = useState(false);
    const [selectedAnnonce, setSelectedAnnonce] = useState(null);
    const [deleteAnnonce, { 
        loading: mutationLoading,
        error: mutationError,
        data: deleteResponseData 
    }] = useMutation(DELETE_ANNONCE_MUTATION, {
        errorPolicy: 'all'
    });    
    
    const navigateToDetail  = async (annonce) => {
      await setSelectedAnnonce(annonce);
      setShowModal(true);        
    }

    const closeModal = (e) => setShowModal(false);

    const onDelete = (annonce) => {
      if(window.confirm('Voulez-vous vraiment supprimer l\'annonce ?')){
        deleteAnnonce({ variables : {
          annonceId: GraphQLID.parseValue(annonce._id)
        }})
        .then((response) => {               
          const newAnnonces = data.annonces.filter(a => a._id !== response.data.deleteAnnonce._id);          
          updateQuery(prevAnnonces => ({
            annonces: [
              ...newAnnonces
            ]
          }));
          closeModal();
        });
      } 
    }

    if (loading) return (      
      <p>En cours de chargement...</p>      
    );
    if (error) return `Error! ${error.message}`;          

    return (
      <div>
        <a  
          className="row text-lg-right text-md-right mb-4"
          href="/annonces/create"
        >
          <button className="btn btn-primary">Créer une annonce</button>
        </a>
        <div className="row">
          {data.annonces.map((annonce, index) => {
            return (
              <ListAnnonceItem
                key={index}
                annonce={annonce}
                onClick={(e) => navigateToDetail(annonce)}
              />
            );
          })}
        </div>
        {selectedAnnonce && (
          <div
            tabIndex="-1"
            role="modal fade"
            aria-hidden="true"
            className={`modal ${showModal ? "show" : ""}`}
            style={{
              display: showModal ? "block" : "none",
              backgroundColor: showModal ? "rgba(0, 0,0, 0.8)" : "",
            }}
          >
            <div className="modal-dialog modal-lg">
              <div className="modal-content">
                <div className="modal-header">
                  <h3 className="modal-title">
                    {selectedAnnonce && selectedAnnonce.title}
                    <button
                      type="button"
                      className="close mr-2"
                      data-dismiss="modal"
                      aria-label="Close"
                      onClick={closeModal}
                      style={{
                        position: "absolute",
                        right: 0,
                      }}
                    >
                      <span aria-hidden="true">&times;</span>
                    </button>
                  </h3>
                </div>
                <div className="modal-body">
                  <div className="row ">
                    <div className="col-md-6">
                      <div className="row pl-4 pr-4 row">
                        <img
                          className="card-img-top"
                          src={`${API_URL}/${selectedAnnonce.coverImage}`}
                          style={{ height: "10rem" }}
                          alt="Card image cap"
                        />
                      </div>
                      <div className="pl-4 pr-4 pt-4">
                        <div>Prix: {selectedAnnonce.price} €</div>
                        <div className="mt-2">
                          <span className="badge badge-light mr-2">
                            {selectedAnnonce.type}
                          </span>
                          <span className="badge badge-light">
                            {selectedAnnonce.status}
                          </span>
                        </div>
                      </div>
                    </div>
                    <div className="col-md-6">
                      <span
                        className="badge badge-info"
                        style={{ fontSize: "0.7rem" }}
                      >
                        {selectedAnnonce._id}
                      </span>

                      <p className="pr-4" style={{ fontSize: "0.8rem" }}>
                        {selectedAnnonce.description}
                      </p>
                    </div>
                  </div>
                </div>
                <div className="modal-footer">
                  <button
                    type="button"
                    className="btn btn-danger"
                    onClick={(e) => onDelete(selectedAnnonce)}
                  >
                    {mutationLoading ? (
                      <div className="spinner-border text-light" role="status">
                        <span className="sr-only">Suppression en cours...</span>
                      </div>
                    ) : (
                      "Supprimer"
                    )}
                  </button>
                  <button
                    type="button"
                    className="btn btn-secondary"
                    data-dismiss="modal"
                    onClick={closeModal}
                  >
                    Fermer
                  </button>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    );
}

export default ListAnnonce
