import React, { useState } from 'react';
import { useMutation, gql } from "@apollo/client";

const CREATE_ANNONCE_MUTATION = gql`
  mutation createAnnonceMutation($annonce: AnnonceInput) {
    createAnnonce(annonce: $annonce) {
        _id
        title
        type
        status
        published
        price
        description
        availabilityDate        
    }
  }
`;

function CreateAnnonce() {
    const defaultAnnonce = {
        title: '',
        type: '',
        status: 'available',
        published: false,
        price: 0.0,
        description: '',
        availabilityDate: '',
        coverImage: null,
        images: []
    };    
    const [annonce, setAnnonce] = useState({
        ...defaultAnnonce
    });
    const [createAnnonce, { 
        loading: mutationLoading,
        error: mutationError,
        data 
    }] = useMutation(CREATE_ANNONCE_MUTATION, {
        errorPolicy: 'all'
    });
    
    const onSubmit = e => {
        e.preventDefault();
        createAnnonce({ variables: { annonce: annonce } });        
        setAnnonce(defaultAnnonce);
    }

    const onChange = e => (field) => {
        const target = e.target;
        let value;
        if(target.type === 'checkbox') {
            value = target.checked;
        } else if (target.type === 'number') {
            value = Number(target.value);
        } else if (target.type === 'file') {            
            const { multiple, files: [file] } = target;
            value = multiple ? target.files : file;
        }else {
            value = target.value;
        }
        annonce[`${field}`] = value;

        setAnnonce({
            ...annonce
        });                        
    }

    return (
      <div>
        <form onSubmit={onSubmit}>
          <div className="form-group">
            <label htmlFor="annonce-title">Titre</label>
            <input
              type="text"
              className="form-control"
              id="annonce-title"
              aria-describedby="annonce-title-description"
              placeholder="Titre de l'annonce"
              onChange={(e) => onChange(e)("title")}
              required
            />
            <small
              id="annonce-title-description"
              className="form-text text-muted"
            >
              We'll never share your email with anyone else.
            </small>
          </div>
          <div className="form-group">
            <label htmlFor="annonce-type">Type</label>
            <select
              id="annonce-type"
              className="form-control"
              value={annonce.type}
              onChange={(e) => onChange(e)("type")}
              required
            >
              <option value="rental">Location</option>
              <option value="sale">Vente</option>
            </select>
          </div>
          <div className="form-group">
            <label htmlFor="annonce-price">Prix</label>
            <input
              type="number"
              className="form-control"
              value={annonce.price}
              onChange={(e) => onChange(e)("price")}
              required
            />
          </div>
          <div className="form-group">
            <label htmlFor="annonce-status">Status</label>
            <select
              id="annonce-status"
              className="form-control"
              value={annonce.status}
              onChange={(e) => onChange(e)("status")}
              required
            >
              <option value="available">Disponible</option>
              <option value="pending">En attente</option>
              <option value="sold">Vendu</option>
            </select>
          </div>
          <div className="form-group">
            <label htmlFor="annonce-description">Description</label>
            <textarea
              id="annonce-description"
              className="form-control"
              value={annonce.description}
              onChange={(e) => onChange(e)("description")}
              required
            />
          </div>
          <div className="form-group">
            <label htmlFor="annonce-availability-date">
              Date de disponibilité
            </label>
            <input
              type="date"
              id="annonce-availability-date"
              className="form-control"
              value={annonce.availabilityDate}
              onChange={(e) => onChange(e)("availabilityDate")}
              required
            />
          </div>
          <div className="form-check">
            <input
              type="checkbox"
              id="annonce-published"
              className="form-check-input"
              value={annonce.published}
              onChange={(e) => onChange(e)("published")}
            />
            <label className="form-check-label" htmlFor="annonce-published">
              Publier
            </label>
          </div>
          <div className="form-check">
            <label htmlFor="annonce-coverImage">Image de couverture</label>
            <input
              type="file"
              id="annonce-coverImage"
              className="form-control"
              onChange={(e) => onChange(e)("coverImage")}
            />
          </div>
          <div className="form-check">
            <label htmlFor="annonce-images">Image de description</label>
            <input
              type="file"
              multiple
              id="annonce-images"
              className="form-control"
              onChange={(e) => onChange(e)("images")}
            />
          </div>
          {mutationError && (
            <div className="alert alert-danger" role="alert">
              {mutationError.graphQLErrors[0].message}
            </div>
          )}
          <div className="form-group">
            <button
              type="submit"
              className="btn btn-primary"
              disabled={mutationLoading}
            >
              {mutationLoading ? (
                <div className="spinner-border text-light" role="status">
                  <span className="sr-only">Loading...</span>
                </div>
              ) : (
                "Soumettre"
              )}
            </button>
          </div>
          {mutationLoading && <p>Loading...</p>}
        </form>
      </div>
    );
  }

  export default CreateAnnonce;