import React, { useEffect, useRef, useState } from "react";
import * as d3 from 'd3';

const Line = ({ width, height }: any) => {
    const svgRef: any = useRef<SVGSVGElement>(null);
    const [data, setData] = useState<any[]>([]);
    const [loading, setLoading] = useState<boolean>(true);

    const fetchAveragePriceData = async () => {
        const response = await fetch(`https://localhost/average_price_per_month?page=1`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
            },
        });

        if (!response.ok) return console.log('Data not found !');
        const data = await response.json();
        const res = data['hydra:member'].map((v: any) => {
            const split = v.date.split('T')[0];
            return {
                date: split.substring(0, split.lastIndexOf('-')),
                averagePrice: v.averagePrice
            }
        });
        setData(res);
    }

    useEffect(() => {

        fetchAveragePriceData();
        if (data.length > 0) setLoading(false);
        else setLoading(true);

        if (!loading) {
            const margin = {
                top: 20,
                right: 20,
                bottom: 20,
                left: 20,
            };

            const svg = d3
                .select(svgRef.current)
                .attr("transform", `translate(10, 0)`);

            const everything = svg.selectAll("*");
            everything.remove();

            const X: any = d3
                .scaleBand()
                .domain(data.map((d: any) => d.date))
                .rangeRound([0, width - margin.left - margin.right])
                .padding(0.1);

            svg
                .append('g')
                .attr("orient", "bottom")
                .attr("transform", `translate(0, ${height - margin.top - margin.bottom})`)
                .call(d3.axisBottom(X));

            const yDomain: any[] = d3.extent(data, (d: any) => d.averagePrice);
            const Y: any = d3
                .scaleLinear()
                .domain(yDomain)
                .range([height - margin.top - margin.bottom, 0])
                .nice();

            svg
                .append("g")
                .attr("orient", "left")
                .attr("transform", `translate(0, 0)`)
                .call(d3.axisLeft(Y).ticks(6));

            const line = d3
                .line<{
                    x: string,
                    y: number
                }>()
                .x((d: any) => X(d.date))
                .y((d: any) => Y(d.averagePrice))
                .curve(d3.curveMonotoneX);

            svg
                .append("path")
                .datum(data)
                .join("path")
                .attr("class", "line")
                .attr("fill", "none")
                .attr("stroke", "steelblue")
                .attr("stroke-width", 1.5)
                .attr("d", line);
        }


    }, [loading, height, width, data]);

    return (
        <>
            {loading ?
                <h2>Chargement...</h2>
                :
                <>
                    <h1>Évolutions du prix de vente moyen du mètre carré pour les ventes sur l'année 2021</h1>
                    <svg ref={svgRef} width={width} height={height} />
                </>
            }
        </>
    );
};

export default function TimeSeries(props: any) {

    return (
        <div>
            <Line data={props.data} width={props.width} height={props.height} />
        </div>
    )
}