import React, { useEffect, useRef, useState } from "react";
import * as d3 from 'd3';

/* Just for aestheticism for message */
const timeTranslation = (time: string) => {
    if (time === 'day') return 'jour';
    if (time === 'month') return 'mois';
    if (time === 'year') return 'année';
    return '';
}

const Bar = ({ width, height }: any) => {
    const svgRef: any = useRef<SVGSVGElement>(null);
    const [type, setType] = useState<string>('');
    const [startingDate, setStartingDate] = useState<string>('');
    const [endingDate, setEndingDate] = useState<string>('');
    const [data, setData] = useState<any[]>([]);
    const [loading, setLoading] = useState<boolean>(true);
    const [confirmed, setConfirmed] = useState<boolean>(false);

    const fetchNumberOfSalesData = async () => {
        const response = await fetch(`https://localhost/number_sales_by_date/${type}/${startingDate}/${endingDate}?page=1`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
            },
        });

        if (!response.ok) return console.log('Data not found !');
        const data = await response.json();
        const res = data['hydra:member'].map((v: any) => {
            const split = v.date.split('T')[0];
            return {
                date: type !== 'day' ? split.substring(0, split.lastIndexOf('-')) : split.substring(split.lastIndexOf('-') + 1) ,
                numberSales: v.numberSales
            }
        });
        setData(res);
    }

    useEffect(() => {

        if (confirmed) fetchNumberOfSalesData();
        if (data.length > 0) setLoading(false);
        else setLoading(true);

        if (!loading) {
            const margin = { top: 10, right: 10, bottom: 20, left: 40 };

            const svg = d3
                .select(svgRef.current);

            const everything = svg.selectAll("*");
            everything.remove();

            const xScale: any = d3
                .scaleBand()
                .domain(data.map((d: any) => d.date))
                .range([0, width - margin.left - margin.right]);

            const X = d3
                .axisBottom(xScale);

            svg
                .append('g')
                .attr('class', 'x-axis')
                .attr('transform', `translate(0, ${height - margin.top - margin.bottom})`)
                .call(X);

            const yDomain: any[] = [0, d3.max(data, (d: any) => d.numberSales)];
            const yScale: any = d3
                .scaleLinear()
                .domain(yDomain)
                .range([height - margin.top - margin.bottom, 0])
                .nice();

            const Y = d3
                .axisLeft(yScale)
                .ticks(6)
                .tickSize(-(width - margin.left - margin.right));

            svg
                .append('g')
                .attr('class', 'y-axis')
                .call(Y);

            const colorScale = d3.scaleOrdinal(d3.schemePaired);

            svg
                .selectAll('.bar')
                .data(data)
                .enter()
                .append('rect')
                .attr('class', 'bar')
                .attr('x', (d: any) => xScale(d.date))
                .attr('y', (d: any) => yScale(d.numberSales))
                .attr('height', (d: any) => (height - margin.top - margin.bottom - yScale(d.numberSales)))
                .attr('width', _ => xScale.bandwidth())
                .style('fill', (d: any, i: any) => colorScale(i));

            svg
                .selectAll('.bar-label')
                .data(data)
                .enter()
                .append('text')
                .classed('bar-label', true)
                .attr('x', (d: any) => xScale(d.date) + xScale.bandwidth() / 2)
                .attr('dx', 0)
                .attr('y', (d: any) => yScale(d.numberSales))
                .attr('dy', -6)
                .text((d: any) => d.numberSales)
                .attr("text-anchor", "middle");

            svg
                .select('.x-axis')
                .append('text')
                .attr('x', (width - margin.left - margin.right) / 2)
                .attr('y', 60)
                .attr('fill', '#000')
                .style('font-size', '20px')
                .style('text-anchor', 'middle')
                .text('Date');

            svg
                .select('.y-axis')
                .append('text')
                .attr('x', 0)
                .attr('y', 0)
                .attr('transform', `translate(-50, ${(height - margin.top - margin.bottom) / 2}) rotate(-90)`)
                .attr('fill', '#000')
                .style('font-size', '20px')
                .style('text-anchor', 'middle')
                .text('Nombre de ventes');
        }

    }, []);

    const selectedType = (event: any) => {
        if (!['day', 'month', 'year'].includes(event.target.value)) {
            setType('');
            return;
        };
        setType(event.target.value);
    };

    const selectedStartingDate = (event: any) => {
        setStartingDate(event.target.value);
    }

    const selectedEndingDate = (event: any) => {
        if (!startingDate) return;
        const d1: Date = new Date(startingDate);
        const d2: Date = new Date(event.target.value);
        if (d1 > d2) return alert('La date de départ doit être inférieure ou égale à celle d\'arrivée');
        setEndingDate(event.target.value);
    }

    const confirmChoices = (event: any) => {
        if (type.length > 0 && startingDate.length > 0 && endingDate.length > 0) setConfirmed(true);
        else setConfirmed(false);
    }

    return (
        <>
            <select onChange={selectedType}>
                <option>-- Choisissez entre jour, mois et année --</option>
                <option value={'day'}>Jour</option>
                <option value={'month'}>Mois</option>
                <option value={'year'}>Année</option>
            </select>
            {type.length > 0 ?
                <div>
                    <label>De</label>
                    <input type='date' min="2018-01-01" max="2021-12-31" onChange={selectedStartingDate} />
                    <label>à</label>
                    <input type='date' min="2018-01-01" max="2021-12-31" onChange={selectedEndingDate} />
                </div>
                : null
            }
            <button onClick={confirmChoices}>Confirmer</button>
            {loading ?
                <h2>Chargement...</h2>
                :
                <>
                {
                    data.length === 0 ?
                        <h2>Aucune donnée trouvée !</h2>
                        :
                        <>
                            <h1>Nombre de vente par {timeTranslation(type)} entre le {startingDate} et le {endingDate}</h1>
                            <svg ref={svgRef} width={width} height={height} />
                        </>
                }
                </>
            }
        </>
    );
};

export default function BarChart(props: any) {

    return (
        <div>
            <Bar data={props.data} width={props.width} height={props.height} />
        </div>
    )
}