import React, { useEffect, useRef, useState } from "react";
import * as d3 from 'd3';

const Donut = ({ width, height }: any) => {
    /* Fetch svg reference */
    const svgRef: any = useRef<SVGSVGElement>(null);
    const [year, setYear] = useState<string>('');
    const [data, setData] = useState<any[]>([]);
    const [loading, setLoading] = useState<boolean>(true);
    const [confirmed, setConfirmed] = useState<boolean>(false);

    /* Fetch data according to year */
    const fetchSalesByRegionData = async () => {
        const response = await fetch(`https://localhost/sales_by_region/${year}?page=1`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
            },
        });

        if (!response.ok) return console.log('Data not found !');
        const data = await response.json();
        let sum = 0;
        data['hydra:member'].forEach((element: any) => {
            sum += element.numberSales;
        });
        const res = data['hydra:member'].map((v: any) => {
            return {
                region: v.region,
                salesPercentage: (v.numberSales / sum * 100).toFixed(3),
            }
        });
        setData(res);
    }

    useEffect(() => {

        if (confirmed) fetchSalesByRegionData();
        if (data.length > 0) setLoading(false);
        else setLoading(true);

        if (!loading) {
            const margin = 40;
            const radius = Math.min(width, height) / 2 - margin;

            /* Set svg attributes and append a child to it */
            const svg = d3
                .select(svgRef.current)
                .attr('viewBox', `0 0 ${width / 2} ${width / 2}`)
                .append("g")
                .attr("transform", `translate(${Math.min(width, height) / 2}, ${Math.min(width, height) / 2})`);

            /* Remove previous data */
            const everything = svg.selectAll("*");
            everything.remove();

            /* Create the pie */
            const pie: any = d3
                .pie()
                .value((d: any) => d.salesPercentage)

            /* Use data values */
            const ready = pie(data);

            /* Generate random colors stored in array with same size as data */
            const colors: any = d3
                .scaleOrdinal()
                .range(data.map((v: any) => '#' + Math.random().toString(16).slice(-6)));

            /* Create arc */
            const arc: any = d3
                .arc()
                .innerRadius(120)
                .outerRadius(radius);

            /* Create legend to arc */
            const legend: any = d3
                .arc()
                .innerRadius(radius / 1.75)
                .outerRadius(radius);

            /* Create arc for each data value */
            svg
                .selectAll('parts')
                .data(ready)
                .enter()
                .append('path')
                .attr('d', arc)
                .attr('fill', (d: any, i: number) => colors(i))
                .style("opacity", 0.7);

            /* Apply legend to each arc */
            svg
                .selectAll('slices')
                .data(ready)
                .enter()
                .append('g')
                .attr("transform", d => `translate(${legend.centroid(d)})`)
                .attr("class", 'legend-g')
                .style("user-select", "none")
                .append('text')
                .text((d: any) => `${d.data.region}: ${d.data.salesPercentage}%`)
                .style("text-anchor", "middle")
                .style("font-weight", 700)
                .style("fill", '#222')
                .style("font-size", 14);
        }

    }, []);

    const selectedYear = (event: any) => {
        if (!['2018', '2019', '2020', '2021'].includes(event.target.value)) return;
        setYear(event.target.value);
    };

    const confirmChoice = (event: any) => {
        if (year.length > 0) setConfirmed(true);
        else setConfirmed(false);
    }

    return (
        <>
            <select onChange={selectedYear}>
                <option>-- Choisissez une année --</option>
                <option value={'2018'}>2018</option>
                <option value={'2019'}>2019</option>
                <option value={'2020'}>2020</option>
                <option value={'2021'}>2021</option>
            </select>
            <button onClick={confirmChoice}>Confirmer</button>
            {loading ?
                <h2>Chargement...</h2>
                :
                <>
                    {
                        data.length === 0 ?
                            <h2>Aucune donnée trouvée !</h2>
                            :
                            <>
                                <h1>Répartition des ventes par région sur l'année {year}</h1>
                                <svg ref={svgRef} width={width} height={height} />
                            </>
                    }
                </>
            }
        </>

    );
};

export default function DonutChart(props: any) {

    return (
        <div>
            <Donut width={props.width} height={props.height} />
        </div>
    )
}
